//***********************************************************************
//
// Compute EDCs for submission to Interbull for any number of traits
//   By: Pete Sullivan       Date: July 2020
//
//************************************************************************
/*************************************************************************

Purposes of the Program:
  Compute EDCs for submission to Interbull for MACE, allowing for
  Single or Multiple trait national models, for any number of traits.

General Usage:
  For a complete list of options specify -h on the command line.

Copyright (C) 2020 Lactanet Canada:
  This software is not to be re-distributed without the prior 
  approval of Lactanet Canada.

Warranty/Liability:
  This software is freely available, with NO warranty.

Available on-line:
  https://www.cdn.ca/software/mtedc.html

Please see README.TXT for additional details.

*************************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>
#include <math.h>

/************************* external  MACROS  **************************/
#define REAL double
//#define FLOAT double
#define FLOAT float

#define MONITOR_INVERSIONS
#include "invert_5g.c"

#ifdef IN_DEVELOPMENT
#ifndef RELIABILITY_MODULE
#define RELIABILITY_MODULE
#endif
#endif

/**************************  local  MACROS  ***************************/

#define VERSION "5g"

//**********************
//#define DEBUG
//#define DEBUG_FOP

//#define DEBUG_CUSTOM
//#define TEST_AN 378417
//#define TEST_AN 459296
#define TEST_AN 0
#define TEST_DAM 0
#define TEST_SIRE 0

#define FULL_DEBUG 0
#define MAX_TO_DEBUG 5
#define for_debug_list \
  if (nanml > MAX_TO_DEBUG) n = MAX_TO_DEBUG; else n = nanml;	\
  for (i=1; i<=n; i++)

//---------------------

#define BSZ BUFSIZ

#define PARAMFILE 0
#define PEDFILE 1
#define DATAFILE 2
#define EDCFILE 3
#define LOGFILE 4
#define CHKFILE 5
#define SASFILE 6
#define TRUFILE 7
#define STDOUT 8
#define STDERR 9
#define NFILES 10 /* Number of files to open, as specified above */

#define NPRINT_REC 5 /* Number of data records printed to log file */
#define NPRINT_PED 10 /* Number of pedigree records printed to log file */
#define MAXSIRES 1000 /* maximum sires in a single contemporary group */

#define WT_FMT 6

#define MEAN_VARS 5
#define MEAN_LABELS {"EDC", "NPROG", "EDC-NP", "|EDC-NP|", "EDC/NP"}
#define MEAN_VARS_CG 3
#define MEAN_LABELS_CG {"Animals", "Sires", "MGS" }
#define LEN_LABEL 12
#define LEN_TRAIT 15

#define NPROG 1
#define EFFECTIVE_PROG 2
#define ITB2000 3
#define PGS1 4
#define PGS2 5
#define PGS3 6
#define PGS4 7
#define RELY 8

#define NDAU_NE (wfactor == NPROG)
#define MEYER_NE (wfactor == EFFECTIVE_PROG)

// EDC2000
#define ITB_NE (wfactor == ITB2000)

// EDC2004
#define PGS1_NE (wfactor == PGS1)

// Ro considering sire contribution only
#define PGS2_NE (wfactor == PGS2)

// PGS2 considering M+D/2 for MGS
#define PGS3_NE (wfactor == PGS3)

// Switch from Selection Index to MME methodology
#define MM_NE (wfactor > PGS3)

// EDC2006 (Multi-Trait Prog Absorption)
#define PGS4_NE (wfactor == PGS4)

//#define NEW_ET_ADJ

// Approximate Reliabilities instead of EDCs
#define RELIABILITIES (wfactor == RELY)

#define MALE 1
#define FEMALE 2

#define INDEX(i,j) ((i)*neff2+j)
#define INDEX2(i,j) ((i)*neff2*2+j)
#define BB(i,j) B[(i)*neff2+j]
#define CC(i,j) C[(i)*neff2+j]
#define CCdm(i,j) Cdm[(i)*neff2+j]
#define CCPB(i,j) CPB[(i)*neff2*2+j]
#define CCsp(i,j) Csp[(i)*neff2+j]

#define PRINTINFO \
  tm2 = time( &tm2 ); \
  tmsince1 = tm2 - tm1; \
  tmsince0 = tm2 - tm0; \
  fprintf( stdout, "Elapsed time: %02d:%02d:%02d    Total: %02d:%02d:%02d    ", \
	   tmsince1/3600, (tmsince1%3600)/60, (tmsince1%60),	\
	   tmsince0/3600, (tmsince0%3600)/60, (tmsince0%60) );	\
  sprintf( msg, "echo pid=%d  mem'(sz rss)'=`ps -p %d -o 'sz rssize' | tail -n 1`", getpid(), getpid() ); \
  system( msg ); \
  tm1 = tm2;

//  sprintf( msg, "echo pid=%d  mem'(sz rss)'=`ps -p %d --no-headers -o 'sz rss'`", getpid(), getpid() ); \
  //sleep( 2 );


#ifdef WINDOWS

#define PRINTMEM
#define check_RAM

#else
//#define PRINTMEM PRINTINFO
#define PRINTMEM
#define check_RAM check_VSZ(); check_RSS();

#endif

// NOTES: mET is only used in the outdated Selection Index approach
//        from PGS4/REL4/ACC4 forward '.me' has dimensions ntraits^2 and '.mET' is not used
#define mET me+nmh

#define ET_born (!sire_model && pp->matdam && pp->matdam != pp->dam)


/**************************** STRUCTURES ******************************/

typedef struct {
  int an, dam;
  FLOAT *effdir, *effmat;
} RECORD;

typedef struct PEDIGREE {
  int sire, dam, matdam, nprog, sex, nmates, *mates, nm2, *m2;
  FLOAT *np, *npgp, *nmgp;
  FLOAT *rel, *ne, *me, *edc, *pev, *pev2, *npe, *npe2, *npe3, *medc;
#ifdef RELIABILITY_MODULE
  FLOAT **Xm;
#endif
  struct PEDIGREE **prog;
} PED;

typedef struct {
  int n, m, *rec, *nt;
  FLOAT *wt;
  PED **an, **dam;
} GROUP;

typedef struct {
  int n;
  REAL u, ss, min, max;
} MEAN_STAT;


/************************* GLOBAL VARIABLES ***************************/

RECORD *record, *rr, **pntrec;
MEAN_STAT *means[MEAN_VARS], *means_np[MEAN_VARS], *means_edc[MEAN_VARS], *mstat;
MEAN_STAT *means_cg[MEAN_VARS_CG];
char *mean_label[MEAN_VARS] = MEAN_LABELS;
char *mean_label_cg[MEAN_VARS_CG] = MEAN_LABELS_CG;
char **trait_label, *value[BSZ];
char param_file[BSZ], EDC_method[BSZ*4];
PED *ped, *pp, *ss, *dd, *ds, *mgss, *pgss;
FILE *f[NFILES];
FILE *fped=0, *fpar=0, *frec=0, *fout=0, *flog=0, *fcheck=0, *fsas=0, *ftru=0, *frel=0;
char fnames[NFILES][256], fname[256], fname_stdout[256], fname_stderr[256], version[128];
char line[BSZ], split_up_line[BSZ], msg[BSZ], fmt[BUFSIZ], **index_label;
char cmdline[BSZ], rel_lbl[64];
char matrix_label[4][3], edc_pgm[BSZ], edc_tmp[BSZ], edc_suf[BSZ];
char edc_log[BSZ], edc_par[BSZ], fname_true_rel[256], true_suff[256];
int an, sire, dam, nanml, ncg, nSI, nvalues;
long long nMME, nMMEh;
int ntraits, neffects, neff2, nmaternal, nth, neh, ne2h, noh, nmh, nm2, nEDCh;
int nt2p;
int nsols, nsh, nindex, nEDC, nei, neih, nobs, ndata, nrec, nnn, nflds[NFILES];
int *data, CGvar;
int PEvar, *PEdir, *PEmat, nPEdir, nPEmat, *PEdir_trait, *PEmat_trait;
int *Geq0, nGeq0;
int *include_an, *include_sire, *include_traits;
int allocated[2]={0,0};
REAL *R, *R2, *G, *G2, *S, *RS, *GEDC, *GEDC2, *CG, *E, *E2, *P, *P2, *B, *C, *Cdm, *Csp;
REAL *Pdam, *Pprog, *Psire, *CPC, *CPB, *CPCds, *Rinv_tmp, **Rinv, *combination;
REAL *matrix[4], **weights, **scaled_wts, **trait_wts, *index_h2, *alpha, *galpha;
REAL *observed, *mobserved, *sum_wts;
REAL *lhs, *pev, *recwt, *nu, *vary, *varg;
REAL z1, z2, z4, z8, z16;
int *Rinv_n, **Rinv_ij, *combination_inc, ncomb;
int runcheck=0, redirect_output=0, redirect_stdout=0, redirect_stderr=0;
int sire_model=0, animal_model_variances=0, ignore_h2_check=0;
#ifdef RELIABILITY_MODULE
int wfactor=RELY;
#else
int wfactor=PGS4;
#endif
int ignore_PD_check=0, ignore_MAXSIRES_check=0, ignore_WTS_check=0, ignore_RECWTS=0;
int ignore_SIRExMGS=0, h2_within_CG=0;
int check_SI=0, cancel_check=0, bigCG=0, ignore_dam_rel=0, maternal_only=0;
int call_edc_pgm=0, output_integer=1, maxlen_trait=LEN_LABEL;
int ignore_resid_correlations=0, ignore_ETs=0, HetVar=0, ignore_wts=0;
int n_include_traits=0;
int default_WF=1, file_access_error=0, RAMused[2]={0,0};
int advanced_mate_absorb=0, meyer_absorb=0, fop_quiet=0, edc_reliabilities=0;
int output_accuracies=0;
int count_grandprogeny=0, printing_help=0, TRAITS_given=0, delimited_output=0;
int include_header=0;
time_t tm0, tm1, tm2, tmsince1, tmsince0;
char RSS_tempfile[BUFSIZ];
int Kbyte, Mbyte, Gbyte, RAMmonitor, RAMmonitor_0;

//#define MAXFS 10000   // maximum full-sib family size
int *chron_position, *parents_first, *pstart=0, max_mates=0;

/**************************  PROTOTYPES  ******************************/

void absorb_init();
void absorb_NeMe( PED *sss, PED *ddd );
void accum_mstat();
void accum_Ne();
int check_bit( int i, int flags );
void check_RSS_old();
void check_RSS();
void check_VSZ();
int checkdir( char *fn );
void chron( int an, int gen, int *order, int *list );
char *cfmt( char *cc, int ll );
void close_all_files();
void compute_edcs();
void convert_AMV_to_SMV();
void c_Ro_ITB();
void compute_Ro();
void error( char *mmm );
void ex( int code );
int ET_record();
void f2_CPB( int c0, int ctop, int cp0, int cptop, int bp0, int bptop, int b0, int btop );
void f2_CPC( int c0, int ctop, int p0, int ptop );
FILE *fop( char *fn, char *op );
void form_CPC( ) ;
int including_trait( int t0 );
void *init( char *tmsg, int i, int j );
int invert( double *A, int n );
REAL kbk( REAL *k, REAL *b, int n );
void log_list( int n, char *cc );
void mtedc_2_stedc();
void mtedc_abs_MATES();
void mtedc_abs_ME();
void mtedc_abs_D2S();
void mtedc_abs_P2D( REAL zP, REAL zC, REAL zCPC );
void mtedc_abs_P2S( REAL zP, REAL zC, REAL zCPC );
void mtedc_abs_P2SD();
void mtedc_abs_P( REAL zP, REAL zC, REAL zCPC );
void mtedc_abs_PM( REAL zP, REAL zC, REAL zCPC );
void mtedc_abs_PR();
void mtedc_abs_PROG();
void mtedc_abs_RP();
void mtedc_abs_RCOMBINATION( REAL *v, int nnn, int invert_ne );
void mtedc_abs_RECIPS();
void mtedc_abs_RINV( int invert_ne );
int non_sire( PED *p );
void open_files();
void output_edcs();
void output_rels();
void process_option( char *cc );
void print_help( int code );
int qsort_dam( const void *a, const void *b );
int qsort_record( const void *a, const void *b );
int qsort_sire( const void *a, const void *b );
void read_parameters();
void read_ped();
void split_line( int flds_expected, char *fname, int rec_num );
void split_lineB( int flds_expected, char *fname, int rec_num );
void summarize_means( MEAN_STAT *mm[MEAN_VARS], int mean_vars, char *condition );
void summarize_parameters();
int take_int( char *c, int len );
REAL take_REAL( char *c, int len );
void update_allocated( int bytes );
void warning( char *mmm );
void *zinit( char *tmsg, int i, int j );

void pr( FILE *ff, char *cc, REAL *y, int n );
void flpr( FILE *ff, char *cc, FLOAT *y, int n );
void flpr2( FILE *ff, char *cc, FLOAT *y, int rows, int cols, int nofs );

#ifdef RELIABILITY_MODULE
#include "crEDC_5g_rel.c"
#endif

#ifdef IN_DEVELOPMENT
#include "crEDC_5g_supp.c"
#endif

#ifndef DEBUG_APPROX_REL
#define DEBUG_APPROX_REL 0
#endif

//**********************************************************************
/***************************  SUBROUTINES  ****************************/
//**********************************************************************

void pr( FILE *ff, char *cc, REAL *y, int n ) {
  // write to ff: first 'n' linear elements of 'y' in lower-matrix form
  int i, j, k, m;
  fprintf( ff, "%s: \n", cc );
  for (i=k=0; k<n; i++) {
    if (including_trait(i)) fprintf( ff, "%2d", i );
    for (j=0; j<=i; j++,k++) {
      if (including_trait(i) && including_trait(j)) {
	fprintf( ff, " %8.4f", y[k] );
      }
    }
    if (including_trait(i)) fprintf( ff, "\n" );
  }
  for (i=k=0; k<n; i++) {
    if (including_trait(i)) fprintf( ff, " %8d", i );
    for (j=0; j<=i; j++,k++);
  } fprintf( ff, "\n" );
  for (i=k=m=0; k<n; i++) {
    for (j=0; j<=i; j++,k++) {
      if (!including_trait(i) || !including_trait(j)) if (y[k]) {
	fprintf( ff, "[%d,%d] = %8.4f  ???\n", i, j, y[k] );
	m = 1;
      }
    }
  }
  if (m) {
    // print the full matrix, non-zero values found for excluded trait(s)
    for (i=k=0; k<n; i++) {
      fprintf( ff, "%2d", i );
      for (j=0; j<=i; j++,k++) fprintf( ff, " %8.4f", y[k] );
      fprintf( ff, "\n" );
    }
    for (i=k=0; k<n; i++) {
      fprintf( ff, " %8d", i );
      for (j=0; j<=i; j++,k++);
    } fprintf( ff, "\n" );
  }
  return;
}

void flpr( FILE *ff, char *cc, FLOAT *y, int n ) {
  // write to ff: first 'n' linear elements of 'y' in lower-matrix form
  int i, j, k, m;
  fprintf( ff, "%s: \n", cc );
  for (i=k=0; k<n; i++) {
    if (including_trait(i)) fprintf( ff, "%2d", i );
    for (j=0; j<=i; j++,k++) {
      if (including_trait(i) && including_trait(j)) {
	fprintf( ff, " %8.4f", y[k] );
      }
    }
    if (including_trait(i)) fprintf( ff, "\n" );
  }
  for (i=k=0; k<n; i++) {
    if (including_trait(i)) fprintf( ff, " %8d", i );
    for (j=0; j<=i; j++,k++);
  } fprintf( ff, "\n" );
  fprintf( ff, "\n" );
  for (i=k=m=0; k<n; i++) {
    for (j=0; j<=i; j++,k++) {
      if (!including_trait(i) || !including_trait(j)) if (y[k]) {
	fprintf( ff, "[%d,%d] = %8.4f  ???\n", i, j, y[k] );
	m = 1;
      }
    }
  }
  if (m) {
    // print the full matrix, non-zero values found for excluded trait(s)
    for (i=k=0; k<n; i++) {
      fprintf( ff, "%2d", i );
      for (j=0; j<=i; j++,k++) fprintf( ff, " %8.4f", y[k] );
      fprintf( ff, "\n" );
    }
    for (i=k=0; k<n; i++) {
      fprintf( ff, " %8d", i );
      for (j=0; j<=i; j++,k++);
    } fprintf( ff, "\n" );
  }
  return;
}

void flpr2( FILE *ff, char *cc, FLOAT *y, int rows, int cols, int nofs ) {
  // write to ff: first 'n' linear elements of 'y' in lower-matrix form
  int i, j, k;
  fprintf( ff, "%s: \n", cc );
  for (i=0; i<rows; i++) {
    if (including_trait(i)) fprintf( ff, "%2d", i );
    for (j=0; j<cols; j++) {
      if (including_trait(i) && including_trait(j)) {
	fprintf( ff, " %8.4f", y[i*nofs+j] );
      }
    }
    if (including_trait(i)) fprintf( ff, "\n" );
  }
  for (i=0; i<cols; i++) if (including_trait(i)) fprintf( ff, " %8d", i );
  fprintf( ff, "\n" );
  for (i=k=0; i<rows; i++) {
    for (j=0; j<cols; j++) {
      if (!including_trait(i) || !including_trait(j)) if (y[i*nofs+j]) {
	fprintf( ff, "[%d,%d] = %8.4f ???\n", i, j, y[i*nofs+j] );
	k = 1;
      }
    }
  }
  if (k) {
    // print the full matrix, non-zero values found for excluded trait(s)
    for (i=0; i<rows; i++) {
      fprintf( ff, "%2d", i );
      for (j=0; j<cols; j++) fprintf( ff, " %8.4f", y[i*nofs+j] );
      fprintf( ff, "\n" );
    }
    for (i=0; i<cols; i++) fprintf( ff, " %8d", i );
    fprintf( ff, "\n" );
  }
  return;
}


//**********************************************************************
void absorb_init() {
  // start P and C matrices with the covariance portions
  // effective record information will be added after this function call
  int i, j, ii, jj, id, im, jd, jm;
  for (i=0; i<neh; i++) P[i] = G2[i];
  if (PEvar) {
    for (; i<nsh; i++) P[i] = 0.0;
    for (ii=0; ii<nPEdir; ii++) {
      i = PEdir_trait[ii];
      id = ii + neffects;
      for (jj=0; jj<=ii; jj++) {
	j = PEdir_trait[jj];
	jd = jj + neffects;
	P[ irc(id,jd) ] = E2[ irc(i,j) ];
      }
    }
    for (ii=0; ii<nPEmat; ii++) {
      i = PEmat_trait[ii];
      id = ii + neffects + nPEdir;
      for (jj=0; jj<nPEdir; jj++) {
	j = PEdir_trait[jj];
	jd = jj + neffects;
	P[ irc(id,jd) ] = E2[ irc(i,j) ];
      }
      for (jj=0; jj<=ii; jj++) {
	j = PEmat_trait[jj];
	jd = jj + neffects + nPEdir;
	P[ irc(id,jd) ] = E2[ irc(i,j) ];
      }
    }
  }
  for (i=0; i<nsols; i++) for (j=0; j<nsols; j++) CC(i,j) = 0.0;
  return;
}

//**********************************************************************
void absorb_NeMe( PED *sss, PED *ddd ) {
  // MATE absorption for sire/mgs model (absorb both ways here)
  int i, j, k, ii, jj, id, jd, im, jm, iE, jE;
  for (i=0; i<nmaternal; i++) {
    ii = irc(i,i);
    P[ii] = sqrt( Psire[ii] * Pdam[ii] );
  }
  for (; i<ntraits; i++) {
    ii = irc(i,i);
    P[ii] = Psire[ii];
  }
  mtedc_abs_RCOMBINATION( P, ntraits, 0 );
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      CCdm(i,j) = P[irc(i,j)];
    }
  }

  // absorb sss into ddd
  absorb_init();
  for (i=0; i<nsh; i++) P[i] += sss->ne[i];
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jm = j + ntraits;
      CC(jm,i) = CCdm(i,j);  // target is maternal
    }
    for (jj=0; jj<nPEmat; jj++) {
      j = PEmat_trait[jj];
      jE = jj + neffects + nPEdir;
      CC(jE,i) = CCdm(i,j);
    }
  }
#ifdef SLOW
  invert( P, n );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif

  for (i=0; i<nsh; i++) ddd->npe[i] -= CPC[i];  
  
  // absorb ddd into sss
  absorb_init();
  for (i=0; i<nsh; i++) P[i] += ddd->ne[i];
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jm = j + ntraits;
      CC(i,jm) = CCdm(i,j);  // target is direct
    }
    for (jj=0; jj<nPEmat; jj++) {
      j = PEmat_trait[jj];
      jm = jj + neffects + nPEdir;
      CC(i,jm) = CCdm(i,j);
    }
  }
#ifdef SLOW
  invert( P, n );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
  for (i=0; i<nsh; i++) sss->npe[i] -= CPC[i];  

  // re-initialize for next sss X ddd mating pair
  for (i=0; i<nsols; i++) for (j=0; j<nsols; j++) CCdm(i,j) = 0.0;
  for (i=0; i<nsh; i++) Psire[i] = Pdam[i] = 0.0;
  return;
}

//**********************************************************************
void accum_mstat( MEAN_STAT *m, REAL x ) {
  // update MEAN statistic accumulators
  REAL dev;
  if (m->n) {
    if (x > m->max) { 
      m->max = x;
    }else if (x < m->min) {
      m->min = x;
    }
  }else {
    m->min = m->max = x;
  } 
  m->n++;
  dev = x - m->u;
  m->u += dev / m->n;
  m->ss += ( x - m->u ) * dev;
  return;
}

//**********************************************************************
void accum_Ne() {
  // Sum effective records for all animals (Dir, Mat and DxM)
  //  => Effect record combination matrices for SI methods
  //  => sums of Rinv sub-matrices for MT methods
  int ofs, i, j, k, l, ii, jj, ij, t, n, comb, *R_ij;
  int cg, *cgs, *maxcg, an, sire, dam, a, loc, ns, sires[MAXSIRES];
  REAL *wts, ssum[MAXSIRES], sum, x, zij, zji, *Ri;
  GROUP *group, *gg;
  
  cgs = (int *) init( "cgs", sizeof(int), ntraits );
  maxcg = (int *) zinit( "maxcg", sizeof(int), ntraits );
  wts = (REAL *) init( "wts", sizeof(REAL), ntraits );

  /* determine number of CG effects and required memory */
  if (fgets( line, BSZ, frec ) == NULL) {
    sprintf( msg, "File [%s] is missing or empty => no performance data !!\n",
	     fnames[DATAFILE] );
    error( msg );
  }
  rewind( frec );

  // check if variance scale factors in DATAFILE, and verify correct number of values per line
  split_line( 0, fnames[DATAFILE], 0 );
  n = 1 + (nmaternal>0) + ntraits;
  k = 0;
  if (nvalues == (n+ntraits) && !ignore_RECWTS) {
    // Check for reasonable record weights (<=3.0)
    for (i=0; !k && i<1000 && fgets(line, BUFSIZ, frec); i++) {
      split_line( n+ntraits, fnames[DATAFILE], 1 );
      for (j=0; !k && j<ntraits; j++) k = (atof(value[1+n+j]) > 3.0);
    }
    rewind( frec );
    if (k) warning( "\n!! Record weights > 3 in FOBS seem very high !!\n" );
  }
  if (k || (nvalues != n && nvalues != (n+ntraits))) {
    // File does not look correct.  Provide some guidance.
    warning( "\n!! Fields in data file (FOBS) should include either of the following:\n" );
    if (sire_model) {
      warning( "   sire" );
      if (nmaternal) warning( " mgs" );
    }else {
      warning( "   animal" );
      if (nmaternal) warning( " dam" );
    }
    for (i=0; i<ntraits; i++) {
      sprintf( msg, " CG%d", i+1 );
      warning( msg );
    }
    warning( "\n" );
    if (sire_model) {
      warning( "   sire" );
      if (nmaternal) warning( " mgs" );
    }else {
      warning( "   animal" );
      if (nmaternal) warning( " dam" );
    }
    for (i=0; i<ntraits; i++) {
      sprintf( msg, " CG%d", i+1 );
      warning( msg );
    }
    for (i=0; i<ntraits; i++) {
      sprintf( msg, " wt%d", i+1 );
      warning( msg );
    }
    warning( "\n" );
    warning( "where CG(i) is the contemporary group and wt(i) the record weight for trait i\n" );  
    if (k && !ignore_WTS_check) {
      warning( "\n!! FOBS appears to be incorrectly specified\n" );
      warning( "      If record weights > 3.0 are valid,\n" );
      warning( "         use 'OPTION ignore_WTS_check' to override this program abort\n" );
      error( "" );
    }
  }
  if (nvalues > n) {
    HetVar = 1;
    if (nvalues != (n + ntraits )) {
      // exit with suitable error messages
      split_line( n+ntraits, fnames[DATAFILE], 1 );
    }
    if (ignore_RECWTS) HetVar = 0;
  }else if (nvalues != n) {
    warning( "" );
    // exit with suitable error messages
    split_line( n, fnames[DATAFILE], 1 );
  }
  nflds[DATAFILE] = nvalues;

  nrec = nnn = 0;
  fprintf( flog, "\nFirst %d performance records...\n", NPRINT_REC );
  while( fgets( line, BSZ, frec ) != NULL ) {
    split_line( nflds[DATAFILE], fnames[DATAFILE], ++nnn );
    if (ignore_ETs && ET_record()) continue;
    nrec++;
    if (nrec<NPRINT_REC) fputs( line, flog );
    an = atoi( value[1] );
    if (nmaternal) {
      dam = atoi( value[2] );
      if (sire_model) {
	ped[an].matdam = 0;
      }else if (dam) {
	i = ped[an].matdam;
	if (i && (i != dam)) {
	  sprintf( msg, "multiple maternal dams (%d, %d, ..) for animal (%d)\n", 
		   i, dam, an );
	  error( msg );
	}else {
	  ped[an].matdam = dam;
	}
      }
      ofs = 3;
    }else {
      ofs = 2;
    }
    for (i=0; i<ntraits; i++) {
      cgs[i] = atoi( value[ofs] );
      if (cgs[i] && bigCG) cgs[i] = 1;
      ofs += 1;
      if (cgs[i] > maxcg[i]) maxcg[i] = cgs[i];
    }
  }
  fprintf( flog, "\n%d performance records read from %s\n", nrec, fnames[DATAFILE] );
  //  fprintf( flog, "%d contemporary groups\n", ncg );
  rewind( frec );

  ncg = maxcg[0];
  for (i=1; i<ntraits; i++) {
    if (maxcg[i] > ncg) ncg = maxcg[i];
  }
  group = (GROUP *) zinit( "group", sizeof(GROUP), ncg+1 );
  record = (RECORD *) init( "record", sizeof(RECORD), nrec );
  for (i=0; i<nrec; i++) {
    sprintf( msg, "record[%d].effdir", i );
    record[i].effdir = (FLOAT *) zinit( msg, sizeof(FLOAT), ntraits );
  }
  if (nmaternal) {
    for (i=0; i<nrec; i++) {
      sprintf( msg, "record[%d].effmat", i );
      record[i].effmat = (FLOAT *) zinit( msg, sizeof(FLOAT), nmaternal );
    }
  }

  /* store weight, animal and dam by CG, then compute Ne one trait at a time */
  for (t=0; t<ntraits; t++) if (including_trait(t)) {

    fprintf( stdout, "   Trait%3d of%3d  : %s\n", t+1, ntraits, trait_label[t] );
    n = nnn = 0;
    /* determine memory required for current trait */
    while( fgets( line, BSZ, frec ) != NULL ) {
      split_line( nflds[DATAFILE], fnames[DATAFILE], ++nnn );
      if (ignore_ETs && ET_record()) continue;
      if (atoi(value[1]) < 1) continue;
      n++;
      if (nmaternal) {
        ofs = 3;
      }else {
        ofs = 2;
      }
      cg = atoi( value[ofs+t] );
      if (cg && bigCG) cg = 1;
      group[cg].n++;
    }
    rewind( frec );
    check_RAM;
    for (cg=1; cg<=ncg; cg++) {
      gg = group+cg;
      if (t) {
        if (gg->n > gg->m) {
          gg->m = gg->n;
          free( gg->an );
          free( gg->dam );
          free( gg->wt );
          free( gg->rec );
          gg->an = (PED **) init( "gg->an", sizeof(PED *), gg->m );
          gg->dam = (PED **) init( "gg->dam", sizeof(PED *), gg->m );
          gg->wt = (FLOAT *) init( "gg->wt", sizeof(FLOAT), gg->m );
          gg->rec = (int *) init( "gg->rec", sizeof(int), gg->m );
        }
      }else if (gg->n) {
        gg->m = gg->n;
        gg->an = (PED **) init( "gg->an", sizeof(PED *), gg->m );
        gg->dam = (PED **) init( "gg->dam", sizeof(PED *), gg->m );
        gg->wt = (FLOAT *) init( "gg->wt", sizeof(FLOAT), gg->m );
        gg->rec = (int *) init( "gg->rec", sizeof(int), gg->m );
      }
      gg->n = 0;
    }

    /* store data */
    n = nnn = 0;
    while( fgets( line, BSZ, frec ) != NULL ) {
      split_line( nflds[DATAFILE], fnames[DATAFILE], ++nnn );
      if (ignore_ETs && ET_record()) continue;
      if (atoi(value[1]) < 1) continue;
      dam = 0;
      record[n].an = an = atoi( value[1] );
      ofs = 2;
      if (nmaternal) {
	record[n].dam = dam = atoi( value[ofs] );
        ofs += 1;
      }
      ofs += t;
      cg = atoi( value[ofs] );
      if (cg && bigCG) cg = 1;
      if (cg) {
	gg = group+cg;
	gg->an[gg->n] = ped+an;
	gg->dam[gg->n] = ped+dam;
	ofs += ntraits;
	if (!ignore_wts && HetVar) {
	  gg->wt[gg->n] = atof( value[ofs] );
	}else {
	  gg->wt[gg->n] = 1;
	}
        gg->rec[gg->n] = n;
	gg->n++;
      }
      n++;
    }
    rewind( frec );

    /* compute Ne */
    for (cg=1; cg<=ncg; cg++) {
      gg = group+cg;
      ns = 0;
      if (!check_SI) {
	if (CGvar) {
	  sum = CG[irc(t,t)];
	}else {
	  sum = 0;
	}
	for (a=0; a<gg->n; a++) {
	  if (sire_model) {
	    sire = gg->an[a]-ped;
	  }else {
	    pp = gg->an[a];
	    sire = pp->sire;
	  }
	  if (sire) {
	    loc = 0;
	    while( loc<ns && sire != sires[loc] ) loc++;
	    if (loc == ns) {
	      if ((ns == MAXSIRES) && !(ignore_MAXSIRES_check)) {
		sprintf( msg, "More than %d sires were found in CG %d Trait %d.\n",
			 MAXSIRES, cg, t+1 );
		warning( msg );
		warning( "  This is very unusual to have such a large CG.\n" );
		warning( "  Please check the CG codes in your data.\n" );
		warning( "  IF your CG codes are correct:\n " );
		warning( "    Increase the value of MAXSIRES and re-compile the program.\n" );
		error( "    OR add OPTION ignore_MAXSIRES_check in the parameter file.\n");
	      }
	      if (loc < MAXSIRES) {
		sires[loc] = sire;
		ssum[loc] = 0.0;
		ns++;
	      }
	    }
	    if (loc < MAXSIRES) ssum[loc] += gg->wt[a];
	  }
	  sum += gg->wt[a];
	}
	if ((x = gg->n)) accum_mstat( means_cg[0]+t, x );
	if ((x = ns)) accum_mstat( means_cg[1]+t, x );
	
	if (ns < MAXSIRES) {
	  for (a=0; a<ns; a++) {
	    //	    printf( "cg=%d  sire=%d  sum=%.1f  ssum[%d]=%.1f\n", 
	    //    cg, sires[a], sum, a, ssum[a] );
	    if (sum) {
	      ssum[a] = (sum - ssum[a]) / sum;
	    }else {
	      ssum[a] = 0.0;
	    }
	  }
	}
      }
      for (a=0; a<gg->n; a++) {
        pp = gg->an[a];
	if (check_SI || ns == MAXSIRES) {
          record[gg->rec[a]].effdir[t] = 1;
        }else if (MEYER_NE) {
          record[gg->rec[a]].effdir[t] = gg->wt[a] * (sum - gg->wt[a]) / sum;
        }else {
          if (sire_model) {
            sire = pp-ped;          
          }else {
  	    sire = pp->sire;
  	  }
          if (sire) {
            loc = 0;
	    while( loc<ns && sire != sires[loc] ) loc++;
            record[gg->rec[a]].effdir[t] = gg->wt[a] * ssum[loc];
          }else {
            record[gg->rec[a]].effdir[t] = gg->wt[a] * (sum - gg->wt[a]) / sum;
          }
	  //	  printf( "sire=%d  Ne=%.4f\n", sire, record[gg->rec[a]].effdir[t] );
        }
      }
    }

    if (t<nmaternal) {
      /* compute me and met (Ne-maternal) */
      for (cg=1; cg<=ncg; cg++) {
	gg = group+cg;
	ns = 0;
	if (!check_SI) {
	  if (CGvar) {
	    sum = CG[irc(t,t)];
	  }else {
	    sum = 0;
	  }
	  for (a=0; a<gg->n; a++) {
	    if (sire_model) {
	      sire = gg->dam[a]-ped;
	    }else {
	      pp = gg->dam[a];
	      sire = pp->sire;
	    }
	    if (sire) {
	      loc = 0;
	      while( loc<ns && sire != sires[loc] ) loc++;
	      if (loc == ns) {
		if ((ns == MAXSIRES) && !(ignore_MAXSIRES_check)) {
		  sprintf( msg, "More than %d maternal grandsires in CG %d Trait %d.\n",
			   MAXSIRES, cg, t+1 );
		  warning( msg );
		  warning( "  This is very unusual to have such a large CG.\n" );
		  warning( "  Please check the CG codes in your data.\n" );
		  warning( "  IF your CG codes are correct:\n " );
		  warning( "    Increase the value of MAXSIRES and re-compile the program.\n" );
		  error( "    OR add OPTION ignore_MAXSIRES_check in the parameter file.\n");
		}
		if (loc < MAXSIRES) {
		  sires[loc] = sire;
		  ssum[loc] = 0.0;
		  ns++;
		}
	      }
	      if (loc < MAXSIRES) ssum[loc] += gg->wt[a];
	    }
	    sum += gg->wt[a];
	  }
	  if ((x = ns)) accum_mstat( means_cg[2]+t, x );
	  
	  if (ns < MAXSIRES) {
	    for (a=0; a<ns; a++) {
	      //	    printf( "cg=%d  mgs=%d  sum=%.1f  mgssum[%d]=%.1f\n", 
	      //    cg, sires[a], sum, a, ssum[a] );
	      if (sum) {
		ssum[a] = (sum - ssum[a]) / sum;
	      }else {
		ssum[a] = 0.0;
	      }
	    }
	  }
	}
	for (a=0; a<gg->n; a++) {
	  pp = gg->dam[a];
	  if (check_SI || ns == MAXSIRES) {
	    record[gg->rec[a]].effmat[t] = 1;
	  }else if (MEYER_NE) {
            record[gg->rec[a]].effmat[t] = gg->wt[a] * (sum - gg->wt[a]) / sum;
          }else {
            if (sire_model) {
              sire = pp-ped;
            }else {
              sire = pp->sire;
            }
	    if (sire) {
              loc = 0;
	      while( loc<ns && sire != sires[loc] ) loc++;
              record[gg->rec[a]].effmat[t] = gg->wt[a] * ssum[loc];
            }else {
              record[gg->rec[a]].effmat[t] = gg->wt[a] * (sum - gg->wt[a]) / sum;
            }
	    //	    printf( "mgs=%d  Me=%.4f\n", sire, record[gg->rec[a]].effmat[t] );
	    // Force Ne==Me else D<->M absorbs do not work properly
	    x = 0.5 * (record[gg->rec[a]].effdir[t] + 
		       record[gg->rec[a]].effmat[t]);
	    record[gg->rec[a]].effdir[t] = record[gg->rec[a]].effmat[t] = x;
          }
        }
      }
    }

  }

  check_RAM;
  for (cg=1; cg<=ncg; cg++) {
    gg = group+cg;
    if (gg->m) {
      free( gg->an );
      free( gg->dam );
      free( gg->wt );
      free( gg->rec );
    }
  }
  free( group );

  // compute matrix of weights for matrix R in V(y) for each animal (Sel Index)
  //   OR Z'R^Z (MM approaches)
  for (i=0; i<=nanml; i++) {
    ped[i].ne = (FLOAT *) zinit( "ped[i].ne", sizeof(FLOAT), nsh );
    ped[i].me = (FLOAT *) zinit( "ped[i].me", sizeof(FLOAT), nth*2 );
    if (PGS4_NE || sire_model || (RELIABILITIES && !meyer_absorb)) {
      ped[i].npe = (FLOAT *) zinit( "ped[i].npe", sizeof(FLOAT), nsh );
    }
  }

  if (MM_NE) {  // Mixed Model Approach
    for (t=0,rr=record; t<nrec; t++,rr++) {
      // determine the trait combination
      for (i=comb=0; i<ntraits; i++) if (rr->effdir[i]) comb |= (1<<i);
      Ri = Rinv[comb];
      R_ij = Rinv_ij[comb];
      n = Rinv_n[comb];
      // Direct x Direct (half-stored)
      for (i=ij=0; i<ntraits; i++) {
	for (j=0; j<=i; j++,ij++) P[ij] = sqrt( rr->effdir[i] * rr->effdir[j] );
      }
      for (k=0; k<n; k++) {
	ij = R_ij[k];
	ped[rr->an].ne[ij] += P[ij] * Ri[k];
      }
      if (nmaternal) {
	// Direct x Maternal (full-stored)
	for (i=0; i<ntraits; i++) {
	  for (j=0; j<nmaternal; j++) {
	    ij = irc(i,j);
	    P[ij] = sqrt( rr->effdir[i] * rr->effmat[j] );
	  }
	}
	for (k=0; k<n; k++) {
	  ij = R_ij[k];
	  from_irc( ij, &i, &j );
	  if (j<nmaternal) {
	    ped[rr->an].me[i*ntraits+j] += P[ij] * Ri[k];
	  }
	  if (i<nmaternal && i != j) {
	    ped[rr->an].me[j*ntraits+i] += P[ij] * Ri[k];
	  }
	}
	// Maternal x Maternal (half-stored)
	for (i=ij=0; i<nmaternal; i++) {
	  for (j=0; j<=i; j++,ij++) {
	    P[ij] = sqrt( rr->effmat[i] * rr->effmat[j] );
	  }
	}
	for (; ij<nth; ij++) P[ij] = 0.0;
	for (k=0; k<n; k++) {
	  ij = R_ij[k];
	  from_irc( ij, &i, &j );
	  if (i<nmaternal && j<nmaternal) {
	    ped[rr->dam].ne[irc(ntraits+i,ntraits+j)] += P[ij] * Ri[k];
	  }
	}
      }
    }
  }else {  // Selection Index approach
    for (t=0,rr=record; t<nrec; t++,rr++) {
      for (i=ij=0; i<ntraits; i++) {
	for (j=0; j<=i; j++,ij++) {
	  if ((x = rr->effdir[i] * rr->effdir[j])) {
	    ped[rr->an].ne[ij] += sqrt(x);
	  }
	}
      }
      for (i=ij=0; i<nmaternal; i++) {
	for (j=0; j<=i; j++,ij++) {
	  if ((x = rr->effmat[i] * rr->effmat[j])) {
	    zij = rr->effdir[i] * rr->effmat[j];
	    zji = rr->effdir[j] * rr->effmat[i];
	    if (PGS4_NE || ped[rr->an].dam == rr->dam || RELIABILITIES || sire_model) {
	      // with PGS4_NE met is not separated from me
	      ped[rr->dam].me[ij] += sqrt(x);
	      ped[rr->an].ne[nth+i*nmaternal+j] += sqrt(zij);
	      if (j<i) ped[rr->an].ne[nth+j*nmaternal+i] += sqrt(zji);
	    }else {
	      /* ET-born */
	      (ped[rr->dam].mET)[ij] += sqrt(x);
	      ped[rr->an].ne[nth+i*nmaternal+j] += sqrt(zij);
	      if (j<i) ped[rr->an].ne[nth+j*nmaternal+i] += sqrt(zji);
	    }
	  }
	}
      }
    }
  }
#ifdef DEBUG
      if (FULL_DEBUG) {
	for_debug_list {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
      }else {
	if ((i=TEST_AN)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
	if ((i=TEST_SIRE)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
	if ((i=TEST_DAM)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
	if ((i=TEST_AN)) {
	  sprintf( msg, " %d) me", i );
	  flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
	}
      }
#endif
  if (nrec && sire_model && (PGS4_NE || RELIABILITIES)) {
    //printf( "abs_RINV\n" );
    //mtedc_abs_RINV( 0 );
    printf( "abs_ME\n" );
    mtedc_abs_ME();
    if (nmaternal && !ignore_SIRExMGS) {
      // for MACE EDCs => no progeny absorptions under a sire model
      // Go right to MATE absorption, assuming SIRE/MGS are unrelated (i.e. ignore inbreeding)
      // For reliabilities, can still do MATE absorption now, then PROGENY, then PARENTS
      for (pp=ped+1; pp<=ped+nanml; pp++) {
	for (i=0; i<nsh; i++) pp->npe[i] = pp->ne[i];
      }
      pntrec = (RECORD **) init( "pntrec", sizeof(RECORD *), nrec );
      for (i=0; i<nrec; i++) pntrec[i] = record+i;
      qsort( (void *) pntrec, (size_t) nrec, sizeof(RECORD *), qsort_record );
      
      for (i=0; i<nsols; i++) for (j=0; j<nsols; j++) CCdm(i,j) = 0.0;
      for (i=0; i<nsh; i++) Psire[i] = Pdam[i] = 0.0;
      for (t=0; t<nrec; t++) {
	rr = pntrec[t];
	if (t && ((rr->an != an) || (rr->dam != dam)) ) {
	  if (an && dam) {
	    absorb_NeMe( ped+an, ped+dam );
	  }
	}
	an = rr->an;
	for (i=0; i<ntraits; i++) {
	  if ((zij = rr->effdir[i])) Psire[irc(i,i)] += zij;
	}
	dam = rr->dam;
	for (i=0; i<nmaternal; i++) {
	  if ((zij = rr->effmat[i])) Pdam[irc(i,i)] += zij;
	}
      }
      if (an && dam) absorb_NeMe( ped+an, ped+dam ); // the last S-MGS combination
#ifdef DEBUG
      if (FULL_DEBUG) {
	for_debug_list {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	}
	for_debug_list {
	  if (ped[i].npe != ped[i].ne) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }
	}
      }else {
	if ((i=TEST_AN)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  if (ped[i].npe != ped[i].ne) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }
	}
	if ((i=TEST_SIRE)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  if (ped[i].npe != ped[i].ne) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }
	}
	if ((i=TEST_DAM)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  if (ped[i].npe != ped[i].ne) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }
	}
      }
#endif
    }
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      for (i=0; i<nsh; i++) pp->ne[i] = pp->npe[i];
    }

  }else if (!MM_NE) {

    check_RAM;
    rr = record;
    for (rr=record; rr<record+nrec; rr++) free( rr->effdir );
    if (nmaternal) {
      for (rr=record; rr<record+nrec; rr++) free( rr->effmat );
    }
    free( record );
    
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      /* first invert diagonals to minimize divisions */
      for (i=0; i<ntraits; i++) {
	ii = irc(i,i);
	if (pp->ne[ii]) pp->ne[ii] = 1.0 / pp->ne[ii];
      }
      for (i=0; i<nmaternal; i++) {
	ii = nth + i*nmaternal+i;
	if (pp->ne[ii]) pp->ne[ii] = 1.0 / pp->ne[ii];
      }
      for (i=0; i<nmaternal; i++) {
	ii = irc(i,i);
	if (pp->me[ii]) pp->me[ii] = 1.0 / pp->me[ii];
	if (!PGS4_NE && !RELIABILITIES && !sire_model && (pp->mET)[ii]) {
	  (pp->mET)[ii] = 1.0 / (pp->mET)[ii];
	}
      }
      
      /* now multiply off-diagonals by inverted diagonals */
      if (ITB_NE || MEYER_NE) {
	for (i=0; i<ntraits; i++) {
	  for (j=0; j<i; j++) {
	    if (pp->ne[irc(i,i)] && pp->ne[irc(j,j)]) pp->ne[irc(i,j)] = 1.0;
	  }
	}
	for (i=0; i<nmaternal; i++) {
	  for (j=0; j<nmaternal; j++) {
	    if (pp->ne[nth+i*nmaternal+i] && pp->ne[nth+j*nmaternal+j]) 
	      pp->ne[nth+i*nmaternal+j] = 1.0;
	  }
	}
	for (i=0; i<nmaternal; i++) {
	  for (j=0; j<i; j++) {
	    if (pp->me[irc(i,i)] && pp->me[irc(j,j)]) pp->me[irc(i,j)] = 1.0;
	    if ((pp->mET)[irc(i,i)] && (pp->mET)[irc(j,j)]) (pp->mET)[irc(i,j)] = 1.0;
	  }
	}
      }else {
	ij = 0;
	for (i=0; i<ntraits; i++) {
	  ii = irc(i,i);
	  for (j=0; j<i; j++) {
	    jj = irc(j,j);
	    if( pp->ne[ij] ) {
	      pp->ne[ij] *= ( pp->ne[ii] * pp->ne[jj] );
	    }
	    ij++;
	  }
	  ij++;
	}
	for (i=0; i<nmaternal; i++) {
	  for (j=0; j<i; j++) {
	    x = pp->ne[nth+i*nmaternal+i] * pp->ne[nth+j*nmaternal+j];
	    if( pp->ne[nth+i*nmaternal+j] ) pp->ne[nth+i*nmaternal+j] *= x;
	    if( pp->ne[nth+j*nmaternal+i] ) pp->ne[nth+j*nmaternal+i] *= x;
	  }
	}
	ij = 0;
	for (i=0; i<nmaternal; i++) {
	  ii = irc(i,i);
	  for (j=0; j<i; j++) {
	    jj = irc(j,j);
	    if( pp->me[ij] ) {
	      pp->me[ij] *= ( pp->me[ii] * pp->me[jj] );
	    }
	    if( !PGS4_NE && !RELIABILITIES && !sire_model && (pp->mET)[ij] ) {
	      (pp->mET)[ij] *= ( (pp->mET)[ii] * (pp->mET)[jj] );
	    }
	    ij++;
	  }
	  ij++;
	}
      }
    }
  }

  return;
}


//**********************************************************************
int check_bit( int i, int flags ) {
  // is bit 'i' set in 'flags'?
  int j;
  j = (1<<i);
  return ((j&flags) == j);
}

//**********************************************************************
void check_RSS_old() {
  // system check the RAM currently allocated to the program
  //   => these Linux options were not recognized by at least 1 Unix system
  FILE *fRSS;
  int RSS;
  sprintf( msg, "echo `ps -p %d -o rssize | tail -n 1` > %s", getpid(), RSS_tempfile );
  system( msg );
  fRSS = fopen( RSS_tempfile, "r" );
  fscanf( fRSS, "%d", &RSS );
  fclose( fRSS );
  sprintf( msg, "rm %s", RSS_tempfile );
  system( msg );
  if (RSS > RAMused[0]) RAMused[0] = RSS;
  return;  
}

//**********************************************************************
void check_RSS() {
  // system check the RAM currently allocated to the program
  //   => using options recognized by both Linux and Unixw
  FILE *fRSS;
  int RSS;
  sprintf( msg, "ps -p %d -o rssize | tail -n 1", getpid() );
  fRSS = popen( msg, "r" );
  fscanf( fRSS, "%d", &RSS );
  pclose( fRSS );
  if (RSS > RAMused[0]) RAMused[0] = RSS;
  return;  
}

//**********************************************************************
void check_VSZ() {
  // system check the RAM currently allocated to the program
  FILE *fVSZ;
  int VSZ, mb;
  sprintf( msg, "ps -p %d -o vsize | tail -n 1", getpid() );
  fVSZ = popen( msg, "r" );
  fscanf( fVSZ, "%d", &VSZ );
  pclose( fVSZ );
  if (VSZ > RAMused[1]) RAMused[1] = VSZ;
  allocated[0] = RAMused[1] / Mbyte;
  allocated[1] = Kbyte * (RAMused[1] % Mbyte);
  return;  
}

//**********************************************************************
char *cfmt( char *cc, int ll ) {
  // center and return 'cc' in a static string of length 'll'
  static char a[BUFSIZ];
  int i, j, k;
  a[ll] = 0;
  i = strlen( cc );
  j = ll - i;
  if (j <= 0) {
    strncpy( a, cc, ll );
  }else {
    j++;  // if odd spaces put extra space to the left of the string
    j /= 2;
    for (k=0; k<j; k++) a[k] = ' ';
    for (j=0; j<i; j++,k++) a[k] = cc[j];
    for (; k<ll; k++) a[k] = ' ';
  }
  return a;
}

//**********************************************************************
int checkdir( char *fn ) {
  char tmp[BUFSIZ];
  // returns 1 if the path already existed or was created successfully
  sprintf( tmp, "mkdir -p $(dirname %s)", fn );
  return (!system( tmp ));
}

//**********************************************************************
void chron( int an, int gen, int *order, int *list ) {
  // recursive function to sort individuals chronologically, parents first
  int i;
  #define MAXGEN 1000
  static int plist[MAXGEN], nan=0;

  if (an) {
    if (order[an]) {
      return;
    }
    if (gen < MAXGEN) {
      plist[gen]=an;
    }else {
      warning( "Pedigree is too extensive for chronological sort routine\n" );
      sprintf( msg, " ... more than %d generations !!!\n", MAXGEN );
      error( msg );
    }
    // check for a pedigree loop
    i = 0;
    while( i<gen && plist[i] != an ) i++;
    if (i<gen) {
      // animal is it's own ancestor, terminate this chron() branch to avoid loop
      warning( "CHRON WARNING: pedigree loop (an->par->grpar...)\n" );
      for (i=0; i<=gen; i++) {
	if (i) warning( " -> " );
	sprintf( msg, "%d", plist[i] );
	warning( msg );
      }
      warning( "\n" );
      return;
    }
  }else {
    return;
  }

  // no errors were found and animal was not previously processed... check its parents

  chron( ped[an].sire, gen+1, order, list );
  chron( ped[an].dam, gen+1, order, list );
  if (ped[an].matdam && ped[an].matdam != ped[an].dam) {
    chron( ped[an].matdam, gen+1, order, list );
  }

  // parents are now in the list, add this animal
  nan++;
  order[an] = nan;
  list[nan] = an;

  return;
}

//**********************************************************************
void close_all_files() {
  //
  int i;
  for (i=0; i<NFILES; i++) {
    if (f[i]) {
      fclose( f[i] );
      f[i] = 0;
    }
  }
  if (redirect_stderr && stderr) fclose( stderr );
  if (redirect_stdout && stdout) fclose( stdout );
  return;
}

//**********************************************************************
void compute_edcs() {
  // main function after input parameters and files have been processed
  // do all of the steps needed to generate EDCs or reliabilities
  int i, j, k, n, p, t, rec, ofs, cg;
  REAL x, z, den;

  /* count number of progeny records */
  nnn = 0;
  while( fgets( line, BSZ, frec ) != NULL ) {
    split_line( nflds[DATAFILE], fnames[DATAFILE], ++nnn );
    if (ignore_ETs && ET_record()) continue;
    if (atoi(value[1]) < 1) continue;
    an = atoi( value[1] );
    if (nmaternal) {
      dam = atoi( value[2] );
    }else {
      dam = 0;
    }
    if (sire_model) {
      ss = ped + an;
      ds = ped + dam;
      mgss = ds; // Assumes no ET calves (genetic dam = maternal dam)
    }else {
      ss = ped + ped[an].sire;
      if (dam) {
        ds = ped + ped[dam].sire;
      }else {
        ds = ped;
      }
      mgss = ped + ped[ ped[an].dam ].sire; // For ET calves, mgss != ds
    }
    pgss = ped + ss->sire;
    if (ss->edc == NULL) {
      ss->edc = (FLOAT *) zinit( "ss->edc", sizeof(FLOAT), nEDC );
      ss->np = (FLOAT *) zinit( "ss->np", sizeof(FLOAT), nEDC );
      if (PGS4_NE || RELIABILITIES) ss->medc = (FLOAT *) zinit( "ss->medc", sizeof(FLOAT), neh );
    }
    if (ds->edc == NULL) {
      ds->edc = (FLOAT *) zinit( "ds->edc", sizeof(FLOAT), nEDC );
      ds->np = (FLOAT *) zinit( "ds->np", sizeof(FLOAT), nEDC );
      if (PGS4_NE || RELIABILITIES) ds->medc = (FLOAT *) zinit( "ds->medc", sizeof(FLOAT), neh );
    }
    if (count_grandprogeny) {
      if (!pgss->npgp) {
	pgss->npgp = (FLOAT *) zinit( "pgs->npgp", sizeof(FLOAT), nEDC );
      }
      if (!mgss->nmgp) {
	mgss->nmgp = (FLOAT *) zinit( "mgs->nmgp", sizeof(FLOAT), nEDC );
      }
    }

    if (nmaternal) {
      ofs = 3;
    }else {
      ofs = 2;
    }
    for (j=0; j<nindex; j++) mobserved[j] = observed[j] = 0;
    for (i=0; i<ntraits; i++) {
      cg = atoi( value[ofs] );
      if (cg) {
        for (j=0; j<nindex; j++) {
          if (weights[j][i]) observed[j] += weights[j][i] / sum_wts[j];
	  if ( (i<nmaternal) && (weights[j][i+ntraits]) ) mobserved[j] += weights[j][i+ntraits] / sum_wts[j];
        }
      }
      ofs += 1;
    }
    for (j=0; j<nindex; j++) {
      if (observed[j]) {
	// accumulate progeny
	if (ss-ped) ss->np[j] += observed[j];
	if (count_grandprogeny) {
	  // accumulate grandprogeny
	  if (pgss-ped) pgss->npgp[j] += observed[j];
	  if (mgss-ped) mgss->nmgp[j] += observed[j];
	}
      }
      if (mobserved[j] && dam) {
	// accumulate progeny
	if (ds-ped) ds->np[j] += mobserved[j];
	if (count_grandprogeny) {
	  // accumulate grandprogeny
	  pgss = ped + ds->sire;
	  mgss = ped + ds->dam;
	  if (pgss-ped) pgss->npgp[j] += mobserved[j];
	  if (mgss-ped) mgss->nmgp[j] += mobserved[j];
	}
      }
    }
  }
  rewind( frec );

  PRINTMEM;

#ifdef DEBUG_CUSTOM
  if (TEST_SIRE && ped[TEST_SIRE].np) {
    printf( "%d np: ", TEST_SIRE );
    for (i=0; i<nEDC; i++) printf( "%7.0f", ped[TEST_SIRE].np[i] );
    printf( "\n" );
  }
#endif

  if (NDAU_NE) {
    /* use ND in place of EDC */
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      if (pp->np != NULL) {
        for (j=0; j<nEDC; j++) pp->edc[j] = pp->np[j];
      }
    }

  }else if (PGS4_NE || RELIABILITIES) {
    if (!sire_model ) {
      /*
#ifdef DEBUG
      if (FULL_DEBUG) {
	for_debug_list {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
      }else {
	if ((i=TEST_AN)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
	if ((i=TEST_SIRE)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
	if ((i=TEST_DAM)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, neh );
	}
      }
#endif
      //fprintf( stdout, "Absorb R-inverse...\n" );
      //mtedc_abs_RINV( 1 );
      */
#ifdef DEBUG
      if (FULL_DEBUG) {
	for_debug_list {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  sprintf( msg, " %d) me", i );
	  flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
	}
      }else {
	if ((i=TEST_AN)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  sprintf( msg, " %d) me", i );
	  flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
	}
	if ((i=TEST_SIRE)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  sprintf( msg, " %d) me", i );
	  flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
	}
	if ((i=TEST_DAM)) {
	  sprintf( msg, " %d) ne", i );
	  flpr( stdout, msg, ped[i].ne, nsh );
	  sprintf( msg, " %d) me", i );
	  flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
	}
      }
#endif
      fprintf( stdout, "Combine ne and me...\n" );
      mtedc_abs_ME();     // combine ne and me into 1 array
      if (nmaternal) {
#ifdef DEBUG
	if (FULL_DEBUG) {
	  for_debug_list {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }
	  }
	  for_debug_list {
	    if (ped[i].ne) {
	      sprintf( msg, " %d) ne", i );
	      flpr( stdout, msg, ped[i].ne, nsh );
	    }
	  }
	}else {
	  if ((i=TEST_AN)) {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }else {
	      sprintf( msg, " %d) ne", i );
	      flpr( stdout, msg, ped[i].ne, nsh );
	    }
	  }
	  if ((i=TEST_SIRE)) {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }else {
	      sprintf( msg, " %d) ne", i );
	      flpr( stdout, msg, ped[i].ne, nsh );
	    }
	  }
	  if ((i=TEST_DAM)) {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }else {
	      sprintf( msg, " %d) ne", i );
	      flpr( stdout, msg, ped[i].ne, nsh );
	    }
	  }
	}
#endif
#ifndef NEW_ET_ADJ
	fprintf( stdout, "Absorb ET info...\n" );
	mtedc_abs_RECIPS();   // absorb ET and recips into each other, one pairing at a time
#endif
      }
      PRINTMEM;
#ifdef DEBUG
      if (FULL_DEBUG) {
	for_debug_list {
	  if (ped[i].npe) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }
	}
	for_debug_list {
	  if (ped[i].ne) {
	    sprintf( msg, " %d) ne", i );
	    flpr( stdout, msg, ped[i].ne, nsh );
	  }
	}
      }else {
	if ((i=TEST_AN)) {
	  if (ped[i].npe) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }else {
	    sprintf( msg, " %d) ne", i );
	    flpr( stdout, msg, ped[i].ne, nsh );
	  }
	}
	if ((i=TEST_SIRE)) {
	  if (ped[i].npe) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }else {
	    sprintf( msg, " %d) ne", i );
	    flpr( stdout, msg, ped[i].ne, nsh );
	  }
	}
	if ((i=TEST_DAM)) {
	  if (ped[i].npe) {
	    sprintf( msg, " %d) npe", i );
	    flpr( stdout, msg, ped[i].npe, nsh );
	  }else {
	    sprintf( msg, " %d) ne", i );
	    flpr( stdout, msg, ped[i].ne, nsh );
	  }
	}
      }
#endif
    }

    fprintf( stdout, "Absorb progeny...\n" );

    // order animals parents before progeny
    parents_first = (int *) zinit( "parents_first", sizeof(int), nanml+1 );
    chron_position = (int *) zinit( "chron_position", sizeof(int), nanml+1 );
    for (p=1; p<=nanml; p++) {
      chron( p, 0, chron_position, parents_first );
    }
    check_RAM;
    free( chron_position );

#ifdef IN_DEVELOPMENT
    if (advanced_mate_absorb) {
      printf( "PROG_hsM\n" );
      mtedc_abs_PROG_hsM();
      printf( "PARENTS_hsM\n" );
      mtrel_abs_PARENTS_hsM();
      printf( "PROG_hsP\n" );
      mtedc_abs_PROG_hsP();
      printf( "PARENTS_hsP\n" );
      mtrel_abs_PARENTS_hsP();
      if (!meyer_absorb) {
	printf( "PROG_fs\n" );
	mtedc_abs_PROG_fs();
      }
    }else {
      mtedc_abs_PROG();   // absorb { all for REL | non-sire for EDC } progeny into sire and genetic dam
    }
#else
    mtedc_abs_PROG();   // absorb { all for REL | non-sire for EDC } progeny into sire and genetic dam
#endif

#ifdef DEBUG
    if (FULL_DEBUG) {
      for_debug_list {
	if (ped[i].npe) {
	  sprintf( msg, " %d) npe", i );
	  flpr( stdout, msg, ped[i].npe, nsh );
	}
      }
    }else {
      if ((i=TEST_AN) && ped[i].npe) {
	sprintf( msg, " %d) npe", i );
	flpr( stdout, msg, ped[i].npe, nsh );
      }
      if ((i=TEST_SIRE) && ped[i].npe) {
	sprintf( msg, " %d) npe", i );
	flpr( stdout, msg, ped[i].npe, nsh );
      }
      if ((i=TEST_DAM) && ped[i].npe) {
	sprintf( msg, " %d) npe", i );
	flpr( stdout, msg, ped[i].npe, nsh );
      }
    }
#endif
    PRINTMEM;
    /*
    if (ped[2].npe) flpr( stdout, "animal 2 npe", ped[2].npe, nsh );
    if (ped[2].npe2) flpr( stdout, "animal 2 npe2", ped[2].npe2, nsh );
    if (ped[2].npe3) flpr( stdout, "animal 2 npe3", ped[2].npe3, nsh );
    if (ped[3].npe) flpr( stdout, "animal 3 npe", ped[3].npe, nsh );
    if (ped[3].npe2) flpr( stdout, "animal 3 npe2", ped[3].npe2, nsh );
    if (ped[3].npe3) flpr( stdout, "animal 3 npe3", ped[3].npe3, nsh );
    */
    fprintf( stdout, "Absorb mates...\n" );
#ifdef RELIABILITY_MODULE
    if (RELIABILITIES) {
      if (sire_model) meyer_absorb = 1;
      if (!meyer_absorb) {
	mtrel_abs_MATES();  // absorb dams into sires and sires into dams
#ifdef DEBUG
	if (FULL_DEBUG) {
	  for_debug_list {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }
	  }
	  for_debug_list {
	    if (ped[i].npe2) {
	      sprintf( msg, " %d) npe2", i );
	      flpr( stdout, msg, ped[i].npe2, nsh );
	    }
	  }
	}else {
	  if ((i=TEST_AN)) {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }
	    if (ped[i].npe2) {
	      sprintf( msg, " %d) npe2", i );
	      flpr( stdout, msg, ped[i].npe2, nsh );
	    }
	  }
	  if ((i=TEST_SIRE)) {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }
	    if (ped[i].npe2) {
	      sprintf( msg, " %d) npe2", i );
	      flpr( stdout, msg, ped[i].npe2, nsh );
	    }
	  }
	  if ((i=TEST_DAM)) {
	    if (ped[i].npe) {
	      sprintf( msg, " %d) npe", i );
	      flpr( stdout, msg, ped[i].npe, nsh );
	    }
	    if (ped[i].npe2) {
	      sprintf( msg, " %d) npe2", i );
	      flpr( stdout, msg, ped[i].npe2, nsh );
	    }
	  }
	}
#endif
      }
      /*
      if (ped[2].npe) flpr( stdout, "animal 2 npe", ped[2].npe, nsh );
      if (ped[2].npe2) flpr( stdout, "animal 2 npe2", ped[2].npe2, nsh );
      if (ped[2].npe3) flpr( stdout, "animal 2 npe3", ped[2].npe3, nsh );
      if (ped[3].npe) flpr( stdout, "animal 3 npe", ped[3].npe, nsh );
      if (ped[3].npe2) flpr( stdout, "animal 3 npe2", ped[3].npe2, nsh );
      if (ped[3].npe3) flpr( stdout, "animal 3 npe3", ped[3].npe3, nsh );
      */
      PRINTMEM;

      /*
      if (ped[1].npe) flpr( stdout, "animal 1 npe", ped[1].npe, nsh );
      if (ped[1].npe2) flpr( stdout, "animal 1 npe2", ped[1].npe2, nsh );
      if (ped[1].npe3) flpr( stdout, "animal 1 npe3", ped[1].npe3, nsh );
      */

      fprintf( stdout, "Absorb parents...\n" );
      mtrel_abs_PARENTS();  // absorb parents into progeny

#ifdef DEBUG
    if (FULL_DEBUG) {
      for_debug_list {
	if (ped[i].npe) {
	  sprintf( msg, " %d) npe", i );
	  flpr( stdout, msg, ped[i].npe, nsh );
	}
      }
      for_debug_list {
	if (ped[i].npe2) {
	  sprintf( msg, " %d) npe2", i );
	  flpr( stdout, msg, ped[i].npe2, nsh );
	}
      }
      for_debug_list {
	if (ped[i].npe3) {
	  sprintf( msg, " %d) npe3", i );
	  flpr( stdout, msg, ped[i].npe3, nsh );
	}
      }
    }else {
      if ((i=TEST_AN)) {
	if (ped[i].npe) {
	  sprintf( msg, " %d) npe", i );
	  flpr( stdout, msg, ped[i].npe, nsh );
	}
	if (ped[i].npe2) {
	  sprintf( msg, " %d) npe2", i );
	  flpr( stdout, msg, ped[i].npe2, nsh );
	}
      }
      if ((i=TEST_SIRE)) {
	if (ped[i].npe) {
	  sprintf( msg, " %d) npe", i );
	  flpr( stdout, msg, ped[i].npe, nsh );
	}
	if (ped[i].npe2) {
	  sprintf( msg, " %d) npe2", i );
	  flpr( stdout, msg, ped[i].npe2, nsh );
	}
      }
      if ((i=TEST_DAM)) {
	if (ped[i].npe) {
	  sprintf( msg, " %d) npe", i );
	  flpr( stdout, msg, ped[i].npe, nsh );
	}
	if (ped[i].npe2) {
	  sprintf( msg, " %d) npe2", i );
	  flpr( stdout, msg, ped[i].npe2, nsh );
	}
      }
    }
#endif

      /*
      if (ped[1].npe) flpr( stdout, "animal 1 npe", ped[1].npe, nsh );
      if (ped[1].npe2) flpr( stdout, "animal 1 npe2", ped[1].npe2, nsh );
      if (ped[1].npe3) flpr( stdout, "animal 1 npe3", ped[1].npe3, nsh );
      if (ped[3].npe) flpr( stdout, "animal 3 npe", ped[3].npe, nsh );
      if (ped[3].npe2) flpr( stdout, "animal 3 npe2", ped[3].npe2, nsh );
      if (ped[3].npe3) flpr( stdout, "animal 3 npe3", ped[3].npe3, nsh );
      */
    }else {
      mtedc_abs_MATES();  // absorb non-sire progeny into sire X mate, then mate into sire
    }
#else
    mtedc_abs_MATES();  // absorb non-sire progeny into sire X mate, then mate into sire
#endif
    PRINTMEM;
    if (RELIABILITIES) {
      output_rels();
    }else {
      fprintf( stdout, "convert mtedc to stedc...\n" );
      mtedc_2_stedc();
    }

  }else {
    /* accumulate EDC for sires */
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      for (j=0; j<nEDC; j++) {
        /* add to own EDC if this animal is a sire */
        if (pp->edc != NULL && pp->rel[j]) {
          pp->edc[j] += (z = alpha[j] * pp->rel[j] / (1.0 - pp->rel[j]));
        }
      }
      if (!sire_model && pp->sire) {
        ss = ped+pp->sire;
        for (j=0; j<nEDC; j++) {
          /* add to sire's EDC */
          if (pp->rel[j]) {
            x = 0;
            if (pp->dam) {
              x = ped[pp->dam].rel[j];
            }
            if (x && !ignore_dam_rel) {
              den = 4 - pp->rel[j] * (1.0 + x);
            }else {
              den = 4 - pp->rel[j];
            }
            
            ss->edc[j] += (z = alpha[j] * pp->rel[j] / den);
#ifdef DEBUG_CUSTOM
	    if (TEST_SIRE && pp->sire == TEST_SIRE) {
	      fprintf( stdout, "Prog %d: trait=%d  rel=%f  contribution=%f\n",
		       pp-ped, j, pp->rel[j], z );
	    }
#endif
          }
        }
      }
    }
  }
  
  return;
}

//**********************************************************************
void convert_AMV_to_SMV() {
  // Generate SIRE-model covariances from input ANIMAL-model matrices
  int i, j, k, ij, jk, im, jm;
  int dd, md, mm;
  ij = 0;
  for (i=0; i<ntraits; i++) {
    for (j=0; j<=i; j++) {
      S[ij] = 0.25 * G[ij];
      ij++;
    }
  }
  for (i=0; i<nmaternal; i++) {
    im = i + ntraits;
    for (j=0; j<=i; j++) {
      jm = j + ntraits;
      dd = irc(i,j);
      md = irc(im,j);
      mm = irc(im,jm);
      if (maternal_only) {
	// MGS maternal contribution
	S[mm] = 0.25 * G[mm];
      }else {
	// MGS total genetic contribution
	S[mm] = 0.25 * (G[mm] + G[md] + 0.25 * G[dd]);
      }
    }
    for (j=0; j<ntraits; j++) {
      jm = j + ntraits;
      md = irc(im,j);
      if (maternal_only) {
	// MGS maternal contribution
	S[md] = 0.25 * G[md];
      }else {
	// MGS total genetic contribution
	S[md] = 0.25 * (G[md] + 0.5 * G[dd]);
      }
    }
  }
  for (i=0; i<neh; i++) G[i] = S[i];
  // Add change to Genetic Covariances to R
  for (i=k=0; i<ntraits; i++) {
    for (j=0; j<=i; j++,k++) {
      RS[k] = G[k];
    }
  }
  for (i=0; i<nmaternal; i++) {
    im = i + ntraits;
    for (j=0; j<=i; j++) {
      jm = j + ntraits;
      ij = irc(im,jm);
      k = irc(i,j);
      RS[k] += G[ij];
      for (jk=0; jk<ntraits; jk++) {
	ij = irc(im,jk);
	k = irc(i,jk);
	RS[k] += G[ij]; /* assumes an-dam relationship = 0.5 */
      }
    }
  }
  for (i=0; i<nth; i++) {
    R[i] += varg[i] - RS[i];
    varg[i] = RS[i];
  }
  return;  
}

//**********************************************************************
void c_Ro_ITB() {
  // Compute Reliability based on an individual's own performance
  //  => covariance matrices described in the method of Interbul, 2000
  int i, j, k, ii, jj, ij, im, iET, jET, imjm, imj, jm, jmjm;
  int dd, dm, md, mm, Ed, Em, EE;
  REAL x, *rel;
  for (i=0; i<nth; i++) {
    P[i] = ( R[i] * pp->ne[i] ) + G[i] + E[i];
  }
  
  if (ET_born) {
    for (i=0; i<nmaternal; i++) {
      im = i+ntraits;
      iET = im+nmaternal;
      /* update direct, maternal, and ET recipient diagonal blocks */
      for (j=0; j<=i; j++) {
	jm = j+ntraits;
	jET = jm+nmaternal;
	dd = irc(i,j);
	mm = irc(im,jm);
	EE = irc(iET,jET);
	P[dd] += E[mm] + G[mm];
	P[mm] = ( (R[dd] + 0.75 * G[dd] + E[dd]) * pp->me[dd] )
	  + 0.25 * G[dd] + G[mm] + E[mm];
	P[EE] = ( (R[dd] + G[dd] + E[dd]) * (pp->mET)[dd] )
	  + G[mm] + E[mm];
      }
      /* assign maternal X direct and ET X direct off-diagonal blocks */
      for (j=0; j<ntraits; j++) {
	dd = irc(i,j);
	md = irc(im,j);
	Ed = irc(iET,j);
	P[md] = 0.5 * G[dd] + G[md];
	P[Ed] = G[md];
      }
      /* assign ET X maternal off-diagonal block */
      for (j=0; j<nmaternal; j++) {
	mm = irc(im,jm);
	Em = irc(iET,jm);
	P[Em] = G[mm];
      }
    }
  }else {
    for (i=0; i<nmaternal; i++) {
      im = i+ntraits;
      iET = im+nmaternal;
      /* update direct, maternal, and ET recipient diagonal blocks */
      for (j=0; j<=i; j++) {
	jm = j+ntraits;
	jET = jm+nmaternal;
	dd = irc(i,j);
	mm = irc(im,jm);
	EE = irc(iET,jET);
	dm = irc(i,jm);
	md = irc(im,j);
	P[dd] += E[mm] + G[mm] + 0.5 * G[md];
	P[mm] = ( (R[dd] + 0.75 * G[dd] + E[dd]) * pp->me[dd] )
	  + 0.25 * G[dd] + G[mm] + 0.5 * (G[dm] + G[md]) + E[mm];
	P[EE] = ( (R[dd] + G[dd] + E[dd]) * (pp->mET)[dd] )
	  + G[mm] + E[mm];
      }
      /* assign maternal X direct and ET X direct off-diagonal blocks */
      for (j=0; j<ntraits; j++) {
	dd = irc(i,j);
	md = irc(im,j);
	Ed = irc(iET,j);
	/* Can use G[md] for G[dm] here because G and P[dd] are symmetric ... */
	P[dd] += 0.5 * G[md];
	P[md] = 0.5 * G[dd] + G[md];
	P[Ed] = G[md];
      }
      for (j=0; j<nmaternal; j++) {
	jm = j+ntraits;
	mm = irc(im,jm);
	dm = irc(i,jm);
	md = irc(im,j);
	Ed = irc(iET,j);
	P[md] += 0.25 * G[dm] + 0.5 * G[mm];
	P[Ed] += 0.5 * G[mm];
      }
      /* assign ET X maternal off-diagonal block */
      for (j=0; j<nmaternal; j++) {
	jm = j+ntraits;
	mm = irc(im,jm);
	md = irc(im,j);
	Em = irc(iET,jm);
	P[Em] = G[mm] + 0.5 * G[md];
      }
    }
  }
  return;
}

//**********************************************************************
void compute_Ro() {
  // Compute Reliability based on an individual's own performance
  // 
  int i, j, k, ii, jj, ij, im, iET, jET, imjm, imj, jm, jmjm;
  int dd, dm, md, mm, Ed, Em, EE;
  REAL x;
  FLOAT *rel;

  for (i=0; i<noh; i++) P[i] = 0.0;

#ifdef DEBUG
  if (FULL_DEBUG || (TEST_SIRE && pp->sire == TEST_SIRE)) {
    if (pp->np) {
      fprintf( flog, "Animal %d", (int) (pp-ped) );
      if (ET_born) fprintf( flog, " (ET born)" );
      fprintf( flog, "\n" );
      fprintf( flog, "  ne: " );
      for (i=0; i<nth; i++) fprintf( flog, "%8.4f", pp->ne[i] );
      fprintf( flog, "\n" ); 
      fprintf( flog, "  me: " );
      for (i=0; i<nmh; i++) fprintf( flog, "%8.4f", pp->me[i] );
      fprintf( flog, "1\n" );
      if (!PGS4_NE && !RELIABILITIES) {
	fprintf( flog, " met: " );
	for (i=0; i<nmh; i++) fprintf( flog, "%8.4f", (pp->mET)[i] );
	fprintf( flog, "\n" ); 
      }
    }
  }
#endif


  if (wfactor < PGS2 || sire_model) {
    c_Ro_ITB();

#ifdef IN_DEVELOPMENT
  }else if (PGS2_NE) {
    c_Ro_PGS2();
  }else if (PGS3_NE) {
    c_Ro_PGS3();
#endif

  }else {
    sprintf( msg, "invalid wfactor option [%d] for compute_Ro\n", wfactor );
    error( msg );
  }
      
  /* zero out rows and columns of P for missing observations */
  for (i=0; i<nobs; i++) observed[i] = 0;
  for (i=0; i<ntraits; i++) {
    if (pp->ne[irc(i,i)]) observed[i] = 1;
  }
  for (i=0; i<nmaternal; i++) {
    if (pp->me[irc(i,i)]) observed[ntraits+i] = 1;
    if ((pp->mET)[irc(i,i)]) observed[neffects+i] = 1;
  }
  ij = 0;
  for (i=0; i<nobs; i++) {
    for (j=0; j<=i; j++) {
      if (!observed[i] || !observed[j]) P[ij] = 0.0;
      ij++;
    }
  }

#ifdef DEBUG
  if (FULL_DEBUG) {
    fprintf( flog, "setting up C...\n" );
  }
#endif

  for (i=0; i<neffects; i++) {
    for (j=0; j<neffects; j++) {
      CC(i,j) = G[irc(i,j)];
    }
  }
  for (i=0; i<nmaternal; i++) {
    im = i + ntraits;
    iET = i + neffects;
    for (j=0; j<neffects; j++) {
      CC(iET,j) = G[irc(im,j)];
    }
  }

  for (i=0; i<nmaternal; i++) {
    im = i + ntraits;
    iET = i + neffects;
    for (j=0; j<nmaternal; j++) {
      jm = j + ntraits;
      CC(im,jm) += 0.5 * G[irc(i,jm)];
    }
  }

  pp->rel = (FLOAT *) zinit( "pp->rel", sizeof(FLOAT), nEDC );

  ndata = 0;
  for (i=0; i<ntraits; i++) {
    if (pp->ne[irc(i,i)]) {
      data[ndata] = i;
      ndata++;
    }
  }
  for (i=0; i<nmaternal; i++) {
    if (pp->me[irc(i,i)]) {
      data[ndata] = i+ntraits;
      ndata++;
    }
  }
  for (i=0; i<nmaternal; i++) {
    if ((pp->mET)[irc(i,i)]) {
      data[ndata] = i+neffects;
      ndata++;
    }
  }

#ifdef DEBUG
  if (FULL_DEBUG || (TEST_SIRE && pp->sire == TEST_SIRE)) {
    fprintf( flog, "[P]\n" );
    for (i=0; i<nobs; i++) {
      for (j=0; j<=i; j++) {
	fprintf( flog, "%9.5f", P[irc(i,j)] );
      }
      fprintf( flog, "\n" );
    }  
    fprintf( flog, "[C]\n" );
    for (i=0; i<nobs; i++) {
      for (j=0; j<neffects; j++) {
	fprintf( flog, "%9.5f", CC(i,j) );
      }
      fprintf( flog, "\n" );
    }  
    fprintf( flog, "[G]\n" );
    for (i=0; i<neffects; i++) {
      for (j=0; j<=i; j++) {
	fprintf( flog, "%9.5f", G[irc(i,j)] );
      }
      fprintf( flog, "\n" );
    }  
  }
#endif

  form_CPC();


#ifdef DEBUG
  if (FULL_DEBUG || (TEST_SIRE && pp->sire == TEST_SIRE)) {
    fprintf( flog, "[CP^C]\n" );
    for (i=0; i<nsols; i++) {
      for (j=0; j<=i; j++) {
	fprintf( flog, "%9.5f", CPC[irc(i,j)] );
      }
      fprintf( flog, "\n" );
    }
  }
#endif
  for (i=0; i<nindex; i++) {
    rel = pp->rel+i;
    *rel = (FLOAT) kbk( weights[i], CPC, neffects );
    if (*rel) *rel /= kbk( weights[i], G, neffects );
  }

#ifdef DEBUG
  if (FULL_DEBUG || (TEST_SIRE && pp->sire == TEST_SIRE)) {
    fprintf( flog, "%d rel: ", (int) (pp-ped) );
    for (i=0; i<nEDC; i++) fprintf( flog, "%7.3f", pp->rel[i] );
    fprintf( flog, "\n" );
  }
#endif
  
  return;
}

//**********************************************************************
void error( char *mmm ) {
  static char msg[BUFSIZ];
  fprintf( stderr, "%s", mmm );
  if (flog) fprintf( flog, "%s", mmm );
#ifndef WINDOWS
  if (*fnames[EDCFILE]) {
    sprintf( msg, "  test -f %s && rm -f %s\n", fnames[EDCFILE], fnames[EDCFILE] );
#ifdef IN_DEVELOPMENT
    if (*fnames[CHKFILE]) {
      sprintf( msg+strlen(msg), "  test -f %s && rm -f %s\n",
	       fnames[CHKFILE], fnames[CHKFILE] );
    }
    if (*fnames[SASFILE]) {
      sprintf( msg+strlen(msg), "  test -f %s && rm -f %s\n",
	       fnames[SASFILE], fnames[SASFILE] );
    }
    if (*fnames[TRUFILE]) {
      sprintf( msg+strlen(msg), "  test -f %s && rm -f %s\n",
	       fnames[TRUFILE], fnames[TRUFILE] );
    }
#endif
    fprintf( stderr, "\nExecuting the following statement(s) (due to ERRORS):\n%s\n\n", msg );
    system( msg );
  }
#endif
  ex(5);
}

//**********************************************************************
void ex( int code ) {
  // exit gracefully, with wrap-up messages, closing files, etc.
#ifndef WINDOWS
  check_RAM;
  if (!printing_help) {
    tm2 = time( &tm2 );
    tmsince0 = tm2 - tm0;
    if (flog) {
      fprintf( flog, "\n" );
      fprintf( flog, "Total RAM used : RSS = %d  VSZ = %d Kb\n",
	       RAMused[0], RAMused[1] );
      fprintf( flog, "Total time (hh:mm:ss): %02ld:%02ld:%02ld\n",
	       tmsince0/3600, (tmsince0%3600)/60, (tmsince0%60) );
    }

    fprintf( stdout, "\n***  " );
    if (allocated[0]) {
      fprintf( stdout, "%.2f Gbytes", 
	       ((double) allocated[0]) + ((double) allocated[1]) / Gbyte );
    }else {
      fprintf( stdout, "%.2f Mbytes", ( (double) allocated[1] / Mbyte ) );
    }
    fprintf( stdout, " of RAM was used." );
    fprintf( stdout, "  ***\n\n" );
    
    fprintf( stdout, "\n" );
    fprintf( stdout, "Total RAM used : RSS = %d  VSZ = %d Kb\n",
	     RAMused[0], RAMused[1] );
    fprintf( stdout, "Total time (hh:mm:ss): %02ld:%02ld:%02ld\n",
	     tmsince0/3600, (tmsince0%3600)/60, (tmsince0%60) );
  }
#endif
  close_all_files();
  /*  
      // the below will not work b/c of freopen(stdout).
      // => need to replace stderr/stdout with use of stde/stdo pointers
  if (redirect_output || redirect_stderr) {
    sprintf( msg, "test -s %s && {\n tail %s\necho 'See file %s for all error messages'\n}", 
	     fnames[STDERR], fnames[STDERR], fnames[STDERR] );
    system( msg );
  }
  */
  exit( code );
}

//**********************************************************************
int ET_record() {
  // check if the record is for an animal conceived by embryo-transfer 
  int result=0, an, dam;
  if (nmaternal) {
    an = atoi( value[1] );
    dam = atoi( value[2] );
    if (dam && dam != ped[an].dam) result = 1;
  }
  return result;
}

//**********************************************************************
void f2_CPB( int c0, int ctop, int cp0, int cptop, int bp0, int bptop, int b0, int btop ) {
  /*
   *  compute A'inv(B)C, given half stored symmetric matrix B
   *    NOTE: the resulting matrix is likely non-symmetric
   *    - USING single row absorptions to avoid matrix inversion
   *    A: n0 X n
   *    B: n X n
   *    C: n1 X n
   */
  int i, j, ni, nj, k, l, diag, ofs, n0, n1, n;
  double w, x, z;
  n0 = ctop - c0;
  n1 = btop - b0;
  n = cptop - cp0;
  if (n != bptop - bp0) {
    sprintf( msg, "  Array dimensions :   C' is %dX%d   B is %dX%d\n", 
	     ctop-c0, cptop-cp0, btop-b0, bptop-bp0 );
    warning( msg );
    error( "ERROR: f2_CPB : cannot form C'PB from incompatible arrays\n" );
  }
  for (i=0; i<n0; i++) {
    for (j=0; j<n1; j++) CCPB(i,j) = 0;
    for (j=0; j<n; j++) CCPB(i,n1+j) = CC(c0+i,cp0+j);
  }
  for (i=0; i<n; i++) {
    ni = n0 + i;
    for (j=0; j<n1; j++) {
      CCPB(ni,j) = BB(b0+j,bp0+i);
    }
    for (j=0; j<i; j++) {
      nj = n1 + j;
      CCPB(ni,nj) = CCPB(nj,ni) = P[irc(i,j)]; // off-diags (i<j)
    }
    nj = n1 + j;
    CCPB(ni,nj) = P[irc(i,j)]; // diag (i=j)
  }
  for (i=n-1; i>=0; i--) {
    // absorb row i
    if ((x=CCPB(n0+i,n1+i))) {
      z = -1.0 / x;
      for (j=0; j<n1+i; j++) {
	if ((x=CCPB(n0+i,j))) {
	  w = z * x;
	  for (l=0; l<n0+i; l++) {
	    CCPB(l,j) += w * CCPB(l,n1+i);
	  }
	}
      }
    }
  }
  //  fprintf( stdout, "\n" );
  for (i=0; i<n1; i++) {
    for (j=0; j<n0; j++) CCPB(i,j) = -CCPB(i,j);
  }

  return;
}

/************************************************************************/
void f2_CPC( int c0, int ctop, int p0, int ptop ) {
  /*
   *  compute C'inv(P)C, given half stored symmetric matrix P
   *    - USING single row absorptions to avoid matrix inversion
   *    C: n0 X n
   *    P: n X n
   */
  int i, j, k, l, diag, ofs, n0, n;
  double w, x, z;

  k = 0;
  for (i=c0; i<ctop; i++) {
    for (j=0; j<=i; j++) { CPC[k] = 0; k++; }
  }
  for (i=p0; i<ptop; i++) {
    for (j=c0; j<ctop; j++) { CPC[k] = CC(j,i); k++; }
    for (j=p0; j<=i; j++) { CPC[k] = P[irc(i,j)]; k++; }
  }
  n0 = ctop - c0;
  n = ptop - p0;

  for (i=n0+n-1; i>=n0; i--) {
    // absorb row i
    diag = i * (i+3) / 2;
    ofs = diag - i;
    k = 0;
    if (CPC[diag]) {
      z = -1.0 / CPC[diag];
      for (j=0; j<i; j++) {
	if (CPC[ofs+j]) {
	  w = z * CPC[ofs+j];
	  for (l=0; l<=j; l++) { CPC[k] += w * CPC[ofs+l]; k++; }
	}else {
	  k += j + 1;
	}
      }
    }
  }
  for (i=0; i<(n0*(n0+1)/2); i++) CPC[i] = -CPC[i];
  /*
  fprintf( flog, "\nCPC[0-2]=%f %f %f  C and P in f2_CPC:\n", CPC[0], CPC[1], CPC[2] );
  for (i=c0; i<ctop; i++) {
    for (j=p0; j<ptop; j++) fprintf( flog, "%8.4f", CC(i,j) );
    fprintf( flog, "\n" );
  }
  for (i=k=p0; i<ptop; i++) {
    for (j=0; j<=i; j++,k++) fprintf( flog, "%8.4f", P[k] );
    fprintf( flog, "\n" );
  }
  */
  return;
}

//**********************************************************************
FILE *fop( char *fn, char *op ) {
  // open a file, or exit gracefully if there is a problem
  FILE *f, *ff=0;
  f = fopen( fn, op );
  if (f) {
    if (!ff) {
      if (redirect_stdout || redirect_output) ff = stdout;
    }
    if (ff && !fop_quiet) {
      fprintf( ff, "File \"%s\" successfully opened (mode=\"%s\")\n", fn, op );
    }
  }else {
    if (*op == 'w') {
      // make sure the directory exists and try again
      if (!checkdir( fn ) || !(f = fopen( fn, op ))) {
	sprintf( msg, "ERROR: fopen( \"%s\", \"%s\" ) failed.\n", fn, op );
	warning( msg );
	file_access_error = 1;
      }
    }
  }
#ifdef DEBUG_FOP
  if (*op == 'w') setbuf( f, 0 );
#endif
  return f;
}

//**********************************************************************
void form_CPC( ) {
  // matrix multiply: C'P^C
  int i, j, k, l, m, ij, kl;
  invert( P, nobs );
#ifdef DEBUG
  if (FULL_DEBUG) {
    fprintf( stdout, "[P]^\n" );
    ij = 0;
    for (i=0; i<nobs; i++) {
      for (j=0; j<=i; j++) {
	fprintf( stdout, "%10.2f", P[ij] );
	ij++;
      }
      fprintf( stdout, "\n" );
    }
  }
#endif
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
#ifdef DEBUG
      if (FULL_DEBUG) {
	fprintf( stdout, "CPC[%d] = 0", ij );
      }
#endif
      CPC[ij] = 0.0;
      for (k=0; k<ndata; k++) {
        for (l=0; l<ndata; l++) {
          kl = irc(data[k],data[l]);
          CPC[ij] += CC(data[k],i) * CC(data[l],j) * P[kl];
#ifdef DEBUG
	  if (FULL_DEBUG) {
	    fprintf( stdout, " %d %d ", data[k], data[l] );
	    fprintf( stdout, " %f %f %f ", CC(data[k],i), CC(data[l],j), P[kl] );
	    fprintf( stdout, " ..%10.5f (%d) ", CPC[ij], kl );
	  }
#endif
        }
      }
      ij++;
#ifdef DEBUG
      if (FULL_DEBUG) {
	fprintf( stdout, "\n" );
      }
#endif
    }
  }
  return;
}

//**********************************************************************
int including_trait( int t0 ) {
  int i, tr;
  if (n_include_traits && t0 < neffects) {
    // check the trait for inclusion
    tr = 1 + ((t0%neffects)%ntraits);  // map solution row t0 to trait tr
    for (i=0; i<n_include_traits && tr != include_traits[i]; i++);
    return (i<n_include_traits);
  }else {
    // all traits are included
    return 1;
  }
}

//**********************************************************************
void *init( char *tmsg, int i, int j ) {
  // allocate RAM
  void *p=0;
  if (!j) j = 1; // avoid allocation request for zero bytes
  if (i*j > 0) {
    p = malloc( i*j );
    if (p == NULL) {
      PRINTMEM;
      sprintf( msg, "insufficient RAM available for %s[%d*%d], program stopped!!\n",
	       tmsg, i, j );
      error( msg );
    }
    update_allocated( i*j );
  }else {
    sprintf( msg, "init 0 bytes for %s !\n", tmsg );
    warning( msg );
  }
  return p;
}

//**********************************************************************
REAL kbk( REAL *k, REAL *b, int n ) {
  // multiply vector 'k' and matrix 'b': k'bk
  int i, j, m;
  REAL x;
  x = 0.0;
  m = 0;
  for (i=0; i<n; i++) {
    for (j=0; j<i; j++) {
      x += k[i] * k[j] * b[m] * 2;
      m++;
    }
    x += k[i] * k[j] * b[m];
    m++;
  }
  return x;
}

//**********************************************************************
void log_list( int n, char *cc ) {
  // for a header row in the log file (e.g. CG1 CG2 ... CGn)
  int i;
  if (n > 4) {
    for (i=0; i<2; i++) fprintf( flog, " %s%d", cc, i+1 );
    fprintf( flog, " ... %s%d", cc, n );
  }else {
    for (i=0; i<n; i++) fprintf( flog, " %s%d", cc, i+1 );
  }
  return;
}

//**********************************************************************
void mtedc_2_stedc() {
  // convert from an EDC matrix to equivalent single-trait EDCs
  //  => absorb correlated information into the single-trait EDCs
  int i, j, k, ii, ij;
  REAL x, z;
#ifdef DEBUG
  //  if (FULL_DEBUG) {
    fprintf( flog, " *** mtedc_2_stedc ***\n");
    pr( flog, "G", G, neh );
    pr( flog, "G2", G2, neh );
    //  }
#endif

  for (pp=ped+1; pp<=ped+nanml; pp++ ) {
    if (pp->edc != NULL) {
      for (i=0; i<neh; i++) P[i] = G2[i] + pp->medc[i];

#ifdef DEBUG
      if (FULL_DEBUG || ((pp-ped) == TEST_SIRE) ||
	  ((pp-ped) == TEST_AN) || ((pp-ped) == TEST_DAM)) {
	sprintf( msg, "Animal %d) P", (int) (pp-ped) );
	pr( flog, msg, P, neh );
	flpr( flog, "   npe", pp->npe, nsh );
	flpr( flog, "  medc", pp->medc, neh );
      }
#endif
      invert( P, neffects );

#ifdef DEBUG
      if (FULL_DEBUG || ((pp-ped) == TEST_SIRE) ||
	  ((pp-ped) == TEST_AN) || ((pp-ped) == TEST_DAM)) {
	sprintf( msg, "Animal %d) P^", (int) (pp-ped) );
	pr( flog, msg, P, neh );
      }
#endif
      for (j=0; j<nindex; j++) {
	pp->edc[j] = 1.0 - ( kbk( weights[j], P, neffects ) /
			     kbk( weights[j], G, neffects ) );
      }

      z = pp->edc[0];
      // convert reliabilities to EDC for the MACE sire model
      for (i=0; i<nEDC; i++) {
	x = pp->edc[i];
	pp->edc[i] = alpha[i] * x / (1.0 - x);
	if (pp->edc[i] < .001 && pp->edc[i] > -.001) pp->edc[i] = 0;
      }
    }
  }
  return;
}

//**********************************************************************
void mtedc_abs_MATES() {
  // for each sire, loop through mates
  // absorb progeny into sire X mate, then mate into sire (input prog ne[])
  int i, j, k, ij, m, n, p, s, nm;
  REAL x, z;

  if (sire_model) {
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      if (pp->ne) {
	for (i=0; i<nsh; i++) pp->npe[i] = pp->ne[i];
      }
    }
  }else {

    // store lists of progeny with both parents known and ne > 0, for each sire
    // N.B. ASSUMPTION: prog should still be NULL from original zinit()
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      if ( non_sire( pp ) && (pp->sire && pp->dam) ) {
	x = 0.0;
	i = 0;
	while (!x && i<neffects) {
	  x += pp->ne[irc(i,i)];
	  i++;
	}
	if (x) {
	  ped[pp->sire].nprog++;
	}
      }
    }
    for (ss=ped+1; ss<=ped+nanml; ss++) {
      if (ss->nprog) {
	if (!pstart && ss->nprog > max_mates) max_mates = ss->nprog;
	ss->prog = (PED **) zinit( "ss->prog", sizeof( PED *), ss->nprog );
	ss->nprog = 0;
      }
    }
    if (!pstart) {
      max_mates++;
      fprintf( stdout, "Allocating RAM for a maximum of %d mates per sire\n", max_mates );
      pstart = (int *) zinit( "pstart", sizeof(int), max_mates );
    }
    for (p=1; p<=nanml; p++) {
      pp = ped + parents_first[p];
      if ( non_sire( pp ) && (pp->sire && pp->dam) ) {
	x = 0.0;
	i = 0;
	while (!x && i<neffects) {
	  x += pp->ne[irc(i,i)];
	  i++;
	}
	if (x) {
	  ss = ped + pp->sire;
	  ss->prog[ ss->nprog ] = pp;
	  ss->nprog++;
	}
      }
    }
    
    for (s=1; s<=nanml; s++) {
      ss = ped + parents_first[s];
      
      if (!(s%100000)) {
	fprintf( stdout, "%9d) %9d (Sire)\n", s, (int) (ss-ped) );
      }
      if (ss->nprog) {
	
	qsort( (void *)ss->prog, (size_t)ss->nprog, sizeof(PED *), qsort_sire );
	// count number of mates and progeny start points for each one
	nm = k = 0;
	for (i=0; i<ss->nprog; i++) {
	  if (ss->prog[i]->dam != k) {
	    k = ss->prog[i]->dam;
	    if (nm < max_mates-1) pstart[nm] = i;
	    nm++;
	  }
	}
	if (nm < max_mates) {
	  pstart[nm] = ss->nprog;
	}else {
	  fprintf( stderr, "Sire %d has more than maximum number of mates ... %d > %d\n",
		   parents_first[s], nm, max_mates-1 );
	  exit(9);
	}
	
	// process one mate at a time
	for (m=0; m<nm; m++) {
	  dd = ped + ss->prog[pstart[m]]->dam;
	  // absorb all full-sib progeny into sire X mate
	  for (i=0; i<nsols; i++) {
	    for (j=0; j<nsols; j++) {
	      CCdm(i,j) = 0.0;
	    }
	  }
	  for (p=pstart[m]; p<pstart[m+1]; p++) {
	    pp = ss->prog[p];
	    mtedc_abs_P2SD();
	    for (i=0; i<nsols; i++) {
	      for (j=0; j<nsols; j++) {
		// CPB is dam X sire, so add transpose to sire X dam
		CCdm(j,i) += CCPB(i,j);
	      }
	    }
	  }
	  
#ifdef DEBUG
	  if (FULL_DEBUG) {
	    sprintf( msg, "sire %d npe before absorbing mate %d",
		     (int) (ss-ped), (int) (dd-ped) );
	    flpr( stdout, msg, ss->npe, nsh );
	  }
#endif
	  // absorb mate into sire
	  mtedc_abs_D2S();
	  for (i=0; i<nsh; i++) ss->npe[i] -= CPC[i];
	  
	  //	printf( "sire %d: npe=%.4f\n", ss-ped, ss->npe[0] );
	  
	}
#ifdef DEBUG
	if (FULL_DEBUG || (ss-ped) == TEST_SIRE) {
	  sprintf( msg, "sire %d npe after absorbing all mates:", (int) (ss-ped) );
	  flpr( stdout, msg, ss->npe, nsh );
	}
#endif
	
      }
    }
  }

  // move npe to medc for sires
  for (ss=ped+1; ss<=ped+nanml; ss++) {
    if (ss->medc != NULL) {
      for (i=0; i<neh; i++) ss->medc[i] = ss->npe[i];
      if (PEvar) {
	// absorb PE effects
	n = nsols - neffects;
	for (i=k=m=0; i<n; i++) {
	  for (j=0; j<=i; j++,k++) {
	    P[k] = ss->npe[irc(i+neffects,j+neffects)];
	  }
	  for (j=0; j<neffects; j++) {
	    CC(j,i) = ss->npe[irc(j,i+neffects)];   // (row = absorb target)
	    if (!m) m = (CC(j,i)!= 0);
	  }
	}
	if (m) {
#ifdef SLOW
	  invert( P, n );
	  f_CPC( 0, neffects, 0, n );
#else
	  f2_CPC( 0, neffects, 0, n );
#endif
	  for (i=0; i<neh; i++) ss->medc[i] -= CPC[i];
	}
      }
    }
  }

  return;
}

//**********************************************************************
void mtedc_abs_ME() {
  // move DxM from ne[] to me[], and move me[] to ne[]
  //   => to switch from SI approach to MT approach
  // for better MT approach accumulate proper ne[] initially and drop this function
  int i, j, ij, im, ii, jj, n;
  REAL x;
  for (pp=ped+1; pp<=ped+nanml; pp++) {
    if (PEvar) {
      for (ii=n=0; !n && ii<nPEdir; ii++) {
	i = PEdir_trait[ii];
	n = (pp->ne[irc(i,i)] != 0.0);
      }
      if (n) {
	for (ii=0; ii<nPEdir; ii++) {
	  i = PEdir_trait[ii];
	  for (jj=0; jj<=ii; jj++) {
	    j = PEdir_trait[jj];
	    pp->ne[irc(ii+neffects,jj+neffects)] = pp->ne[irc(i,j)] + E2[irc(i,j)];
	  }
	  /*
	  // this should be an X dam (not an X an) ... ?
	  for (jj=0; jj<nPEmat; jj++) {
	    j = PEmat_trait[jj];
	    pp->ne[irc(ii+neffects,jj+neffects+nPEdir)] = E2[irc(i,j+ntraits)];
	  }
	  */
	  for (j=0; j<ntraits; j++) {
	    pp->ne[irc(ii+neffects,j)] = pp->ne[irc(i,j)];
	  }
	}
      }
      for (ii=n=0; !n && ii<nPEmat; ii++) {
	i = PEmat_trait[ii];
	n = (pp->ne[irc(i+ntraits,i+ntraits)] != 0.0);
      }
      if (n) {
	for (ii=0; ii<nPEmat; ii++) {
	  i = PEmat_trait[ii];
	  for (jj=0; jj<=ii; jj++) {
	    j = PEmat_trait[jj];
	    pp->ne[irc(ii+neffects+nPEdir,jj+neffects+nPEdir)] = 
	      pp->ne[irc(i+ntraits,j+ntraits)] + E2[irc(i+ntraits,j+ntraits)];
	  }
	  for (j=0; j<nmaternal; j++) {
	    pp->ne[irc(ii+neffects+nPEdir,j+ntraits)] = pp->ne[irc(i+ntraits,j+ntraits)];
	  }
	}
      }
    }
    for (ii=0; ii<nGeq0; ii++) {
      for (j=0; j<nsols; j++) {
	ij = irc( Geq0[ii], j );
	pp->ne[ij] = 0.0;
      }
    }
    if (pp->npe) for (i=0; i<nsh; i++) pp->npe[i] = pp->ne[i];
	
  }
  return;
}

//**********************************************************************
void mtedc_abs_D2S( ) {
  // derive dam contribution to sire (OR sire to dam) (input dam npe[])
  // Cdm summarizes progeny contributions to sire X dam block
  int i, j, k, ij;
  REAL x, z;
  //  if ( non_sire( dd ) && (dd->sire || dd->dam) ) {
  //    mtedc_abs_PARENTS_OF( dd );
  //  }else {
    for (i=0; i<nsh; i++) P[i] = dd->npe[i];
    //  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] += G2[ij];
      ij++;
    }
  }
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) CC(i,j) = CCdm(i,j);
  }
#ifdef SLOW
  invert( P, nsols );
  f_CPC( 0, nsols, 0, nsols );
#else
  /*
	    printf( "  C:\n" );
	    for (i=0; i<neffects; i++) {
	      for (j=0; j<neffects; j++) printf( "%9.5f", CC(i,j) ); printf( "\n" );
	    }
	    pr( stdout, "P", P, nsh );
  */
  f2_CPC( 0, nsols, 0, nsols );
#endif
  return;
}

//**********************************************************************
void mtedc_abs_P2D( REAL zP, REAL zC, REAL zCPC ) {
  //derive progeny contribution to dam (input prog ne[]=D and me[]=DxM)
  int i, j, k, ij, ii, jj, nnn;
  REAL x, z;
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = zP * G2[ij];
      ij++;
    }
    for (j=0; j<neffects; j++) {
      CC(i,j) = zC * G2[irc(i,j)];
    }
  }
  if (non_sire(pp)) {
    for (i=0; i<nsh; i++) P[i] += pp->ne[i];
    nnn = nsols;
  }else {
    // absorb D into M only, by ignoring me and neXme of prog
    for (i=0; i<nth; i++) P[i] += pp->ne[i];
    // also need to absorb contributions from PEdir
    if (PEvar) {
      for (ii=0; ii<nPEdir; ii++) {
	i = ii + neffects;
	for (j=0; j<ntraits; j++) {
	  k = irc(i,j);
	  P[k] += pp->ne[k];
	}
	for (jj=0; jj<=ii; jj++) {
	  j = jj + neffects;
	  k = irc(i,j);
	  P[k] += pp->ne[k];
	}
      }
      for (ii=0; ii<nPEmat; ii++) {
	i = ii + neffects + nPEdir;
	for (jj=0; jj<nPEdir; jj++) {
	  j = jj + neffects;
	  P[irc(i,j)] += E2[ irc( PEmat_trait[ii]+ntraits, PEdir_trait[jj] ) ];
	}
	for (jj=0; jj<=ii; jj++) {
	  j = jj + neffects + nPEdir;
	  P[irc(i,j)] += E2[ irc( PEmat_trait[ii]+ntraits, PEmat_trait[jj]+ntraits ) ];
	}
      }
    }
    nnn = ntraits;
  }
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jj = ntraits + j;
      if (!nGeq0 || ( (G[irc(i,i)] && G[irc(jj,jj)]) )) {
	CC(j+ntraits,i) += pp->me[i*ntraits+j];   // maternal X direct  (row = absorb target = dam)
      }
    }
  }
  if (PEvar) {
    for (ii=0; ii<nPEdir; ii++) {
      i = PEdir_trait[ii];
      for (j=0; j<nmaternal; j++) {
	CC(j+ntraits,ii+neffects) += pp->me[j*ntraits+i];   // maternal X PEdir  (row = absorb target = dam)
      }
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,ii+neffects) += pp->me[j*ntraits+i];   // PEmat X PEdir  (row = absorb target = dam)
      }
    }
    for (i=0; i<ntraits; i++) {
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,i) += pp->me[i*ntraits+j];   // PEmat X direct  (row = absorb target = dam)
      }
    }
  }

  /*
  pr( flog, "inG: ", G2, nsh );
  fprintf( flog, "  C: " );
  for (i=0; i<nsols; i++) {
    fprintf( flog, "\n" );
    for (j=0; j<nsols; j++) fprintf( flog, " %8.4f", CC(i,j) );
  }
  fprintf( flog, "\n" );
  pr( flog, "  P: ", P, nsh );
  */

#ifdef SLOW
  invert( P, nsols );
  //  pr( stdout, "inP: ", P, nsh );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
  for (i=0; i<nsh; i++) CPC[i] = -CPC[i];
  for (i=0; i<neh; i++) CPC[i] += zCPC * G2[i];
  /*
  if (nnn < nsols) {
    for (i=0; i<nsols; i++) {
      for (j=0; j<nnn; j++) CPC[irc(i,j)] = 0;
    }
  }
  */
  return;
}  

//**********************************************************************
void mtedc_abs_P2S( REAL zP, REAL zC, REAL zCPC ) {
  // derive progeny contribution to sire (input prog ne[])
  int i, j, k, ij;
  REAL x, z;
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = zP * G2[ij];
      ij++;
    }
    for (j=0; j<neffects; j++) {
      CC(i,j) = zC * G2[irc(i,j)];
    }
  }
  for (i=0; i<nsh; i++) P[i] += pp->ne[i];
#ifdef SLOW
  invert( P, nsols );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
  for (i=0; i<neh; i++) CPC[i] = zCPC * G2[i] - CPC[i];
  return;
}  

//**********************************************************************
void mtedc_abs_P2SD() {
  //derive progeny contribution to sire X dam (input prog ne[]=D and me[]=DxM)
  static REAL zP=2.0, zC=-1.0, zCPC=0.5;
  int i, j, k, ij, ii, jj;
  REAL x, z;
  //  pr( stdout, "ne", pp->ne, nsh );
  //  if (non_sire(pp)) return;
#ifdef DEBUG
  if (DEBUG_APPROX_REL) {
    pr( stdout, "P2SD-a Psire", Psire, nsh );
  }
#endif
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = BB(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = zP * G2[ij];
      ij++;
    }
    for (j=0; j<neffects; j++) {
      BB(i,j) = CC(i,j) = zC * G2[irc(i,j)];
    }
  }
  if (non_sire(pp)) {
    for (i=0; i<nsh; i++) P[i] += pp->ne[i];
  }else {
    // absorb D into M only, by ignoring me and neXme of prog
    for (i=0; i<nth; i++) P[i] += pp->ne[i];
    // also need to absorb contributions from PEdir
    if (PEvar) {
      for (ii=0; ii<nPEdir; ii++) {
	i = ii + neffects;
	for (j=0; j<ntraits; j++) {
	  k = irc(i,j);
	  P[k] += pp->ne[k];
	}
	for (jj=0; jj<=ii; jj++) {
	  j = jj + neffects;
	  k = irc(i,j);
	  P[k] += pp->ne[k];
	}
      }
      for (ii=0; ii<nPEmat; ii++) {
	i = ii + neffects + nPEdir;
	for (jj=0; jj<nPEdir; jj++) {
	  j = jj + neffects;
	  P[irc(i,j)] += E2[ irc( PEmat_trait[ii]+ntraits, PEdir_trait[jj] ) ];
	}
	for (jj=0; jj<=ii; jj++) {
	  j = jj + neffects + nPEdir;
	  P[irc(i,j)] += E2[ irc( PEmat_trait[ii]+ntraits, PEmat_trait[jj]+ntraits ) ];
	}
      }
    }
  }
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jj = ntraits + j;
      if (!nGeq0 || ( (G[irc(i,i)] && G[irc(jj,jj)]) )) {
	CC(j+ntraits,i) += pp->me[i*ntraits+j];   // maternal X direct  (row = absorb target = dam
      }
    }
  }
  if (PEvar) {
    for (ii=0; ii<nPEdir; ii++) {
      i = PEdir_trait[ii];
      for (j=0; j<nmaternal; j++) {
	CC(j+ntraits,ii+neffects) += pp->me[j*ntraits+i];   // maternal X PEdir  (row = absorb target = dam)
      }
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,ii+neffects) += pp->me[j*ntraits+i];   // PEmat X PEdir  (row = absorb target = dam)
      }
    }
    for (i=0; i<ntraits; i++) {
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,i) += pp->me[i*ntraits+j];   // PEmat X direct  (row = absorb target = dam)
      }
    }
  }
#ifdef SLOW
  invert( P, nsols );
  f_CPB( 0, nsols, 0, nsols, 0, nsols, 0, nsols );
#else
  f2_CPB( 0, nsols, 0, nsols, 0, nsols, 0, nsols );
#endif
#ifdef DEBUG
  if (DEBUG_APPROX_REL) {
    pr( stdout, "P2SD-b Psire", Psire, nsh );
  }
#endif
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CCPB(i,j) = -CCPB(i,j);
    }
  }
  for (i=0; i<neffects; i++) {
    for (j=0; j<neffects; j++) {
      CCPB(i,j) += zCPC * G2[irc(i,j)];  // N.B. contribution to dam X sire
    }
  }
  return;
}

//**********************************************************************
void mtedc_abs_P( REAL zP, REAL zC, REAL zCPC ) {
  // derive progeny contribution to sire (input prog ne[])
  int i, j, k, ij;
  REAL x, z;
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = zP * G2[ij];
      ij++;
    }
    for (j=0; j<neffects; j++) {
      CC(i,j) = zC * G2[irc(i,j)];
    }
  }
  for (i=0; i<nsh; i++) P[i] += pp->ne[i];
#ifdef SLOW
  invert( P, nsols );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
  for (i=0; i<nsh; i++) CPC[i] = -CPC[i];
  for (i=0; i<neh; i++) CPC[i] += zCPC * G2[i];
  //  printf( "abs_P=%f %f %f  prog=%d\n", CPC[0], CPC[1], CPC[2], (int) (pp-ped) );
  return;
}  

//**********************************************************************
void mtedc_abs_PM( REAL zP, REAL zC, REAL zCPC ) {
  //derive progeny contribution to dam (input prog ne[]=D and me[]=DxM)
  int i, j, k, ij, ii, jj, nnn;
  REAL x, z;
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = zP * G2[ij];
      ij++;
    }
    for (j=0; j<neffects; j++) {
      CC(i,j) = zC * G2[irc(i,j)];
    }
  }
  if (non_sire(pp)) {
    for (i=0; i<nsh; i++) P[i] += pp->ne[i];
    nnn = nsols;
  }else {
    // absorb Direct into Maternal only, by ignoring me and neXme of prog
    for (i=0; i<nth; i++) P[i] += pp->ne[i];
    // also need to absorb contributions from PEdir
    if (PEvar) {
      for (ii=0; ii<nPEdir; ii++) {
	i = ii + neffects;
	for (j=0; j<ntraits; j++) {
	  k = irc(i,j);
	  P[k] += pp->ne[k];
	}
	for (jj=0; jj<=ii; jj++) {
	  j = jj + neffects;
	  k = irc(i,j);
	  P[k] += pp->ne[k];
	}
      }
      for (ii=0; ii<nPEmat; ii++) {
	i = ii + neffects + nPEdir;
	for (jj=0; jj<nPEdir; jj++) {
	  j = jj + neffects;
	  P[irc(i,j)] += E2[ irc( PEmat_trait[ii]+ntraits, PEdir_trait[jj] ) ];
	}
	for (jj=0; jj<=ii; jj++) {
	  j = jj + neffects + nPEdir;
	  P[irc(i,j)] += E2[ irc( PEmat_trait[ii]+ntraits, PEmat_trait[jj]+ntraits ) ];
	}
      }
    }
    nnn = ntraits;
  }
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jj = ntraits + j;
      if (!nGeq0 || ( (G[irc(i,i)] && G[irc(jj,jj)]) )) {
	CC(j+ntraits,i) += pp->me[i*ntraits+j];   // maternal X direct  (row = absorb target = dam)
      }
    }
  }
  if (PEvar) {
    for (ii=0; ii<nPEdir; ii++) {
      i = PEdir_trait[ii];
      for (j=0; j<nmaternal; j++) {
	CC(j+ntraits,ii+neffects) += pp->me[j*ntraits+i];   // maternal X PEdir  (row = absorb target = dam)
      }
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,ii+neffects) += pp->me[j*ntraits+i];   // PEmat X PEdir  (row = absorb target = dam)
      }
    }
    for (i=0; i<ntraits; i++) {
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,i) += pp->me[i*ntraits+j];   // PEmat X direct  (row = absorb target = dam)
      }
    }
  }
#ifdef DEBUG
  if (FULL_DEBUG || pp->dam == TEST_DAM) {
    fprintf( stdout, "dam %d for PM (HS) prog absorb:\n", pp->dam );
    pr( stdout, "P", P, nsh );
    fprintf( stdout, "C (MxD)\n" );
    for (i=0; i<nsols; i++) {
      for (j=0; j<nsols; j++) {
	if (including_trait(i) && including_trait(j)) fprintf( stdout, "%7.4f", CC(i,j) );
      }
      if (including_trait(i)) fprintf( stdout, "\n" );
    }
  }
#endif
#ifdef SLOW
  invert( P, nsols );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
  for (i=0; i<nsh; i++) CPC[i] = -CPC[i];
  for (i=0; i<neh; i++) CPC[i] += zCPC * G2[i];

  return;
}  

//**********************************************************************
void mtedc_abs_PR() {
  //derive progeny contribution to ET recip (input prog ne[]=D and me[]=DxM)
  int i, j, k, ij, ii, jj;
  REAL x, z;
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = G2[ij];
      ij++;
    }
  }
  for (i=0; i<nsh; i++) P[i] += pp->ne[i];
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jj = ntraits + j;
      if (!nGeq0 || ( (G[irc(i,i)] && G[irc(jj,jj)]) )) {
	CC(j+ntraits,i) += pp->me[i*ntraits+j];   // maternal X direct  (row = absorb target = recip)
      }
    }
  }
  if (PEvar) {
    for (ii=0; ii<nPEdir; ii++) {
      i = PEdir_trait[ii];
      for (j=0; j<nmaternal; j++) {
	CC(j+ntraits,ii+neffects) += pp->me[j*ntraits+i];   // maternal X PEdir  (row = absorb target = dam)
      }
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,ii+neffects) += pp->me[j*ntraits+i];   // PEmat X PEdir  (row = absorb target = dam)
      }
    }
    for (i=0; i<ntraits; i++) {
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(jj+neffects+nPEdir,i) += pp->me[i*ntraits+j];   // PEmat X direct  (row = absorb target = dam)
      }
    }
  }
#ifdef DEBUG
  if ((pp-ped) == TEST_AN) {
    printf( "abs_PR\n" );
    printf( "  C:\n" );
    for (i=0; i<nsols; i++) {
      for (j=0; j<nsols; j++) printf( "%9.5f", CC(i,j) ); printf( "\n" );
    }
    pr( stdout, "P", P, nsh );
  }
#endif
#ifdef SLOW
  invert( P, nsols );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
#ifdef DEBUG
  if ((pp-ped) == TEST_AN) {
    pr( stdout, "CPC", CPC, nsh );
  }
#endif
  return;
}  

//**********************************************************************
void mtedc_abs_PROG() {
  // chronologically absorb prog ne[] into parent ne[] and parent npe[]
  int i, j, k, l, ij, n, nh, mm, p;
  int done;
  REAL x, z, zAA, zAS, zAD, zSS, zDD, zSD;

#ifdef DEBUG
  if (FULL_DEBUG) {
    fprintf( stdout, "Absorb progeny\n" );
    for_debug_list {
      if (i<=nanml && ped[i].ne) {
	sprintf( msg, "ped[%d].ne", i );
	flpr( stdout, msg, ped[i].ne, nsh );
      }
    }
  }
#endif
  
  if (sire_model) {

    if (RELIABILITIES) {
      for (p=nanml; p>=1; p--) {
	pp = ped + parents_first[p];
	if (!(p%100000)) {
	  fprintf( stdout, "%9d) %9d%9d%9d\n", p, (int) (pp-ped), pp->sire, pp->dam );
	}
#ifdef DEBUG
	if (FULL_DEBUG || 
	    (TEST_AN && (pp-ped) == TEST_AN) ||
	    (TEST_SIRE && (pp->sire) == TEST_SIRE) ||
	    (TEST_DAM && (pp->dam) == TEST_DAM)) {
	  fprintf( stdout, "%9d) %9d%9d%9d [%d]\n", p, (int) (pp-ped), pp->sire, pp->dam, non_sire(pp) );
	  fprintf( stdout, "Absorbing progeny %d\n", (int) (pp-ped) );
	  sprintf( msg, " %d) ne", (int) (pp-ped) );
	  flpr( stdout, msg, pp->ne, nsh );
	}
#endif
	if ( (pp->sire || pp->dam) ) {
	  ss = ped + pp->sire;
	  dd = ped + pp->dam;
	  // progeny contribution is zero unless ne > 0
	  z = 16.0 / (11.0 + 4 * (!pp->sire) + !pp->dam);
	  zAA = z;
	  zAS = -.5 * z;
	  zAD = -.25 * z;
	  zSS = 0.25 * z;
	  zDD = 0.0625 * z;
	  zSD = 0.125 * z;
	  x = 0.0;
	  i = 0;
	  while( !x && i<neffects ) {
	    x += pp->ne[irc(i,i)];
	    i++;
	  }
	  
	  if (x) {
	    
	    // ne > 0 ... absorb the progeny
	    if (pp->dam) {
	      // absorb progeny into mgs, prep for progeny absorption
	      // assume sire unknown (zero contribution to dam/mgs)
	      mtedc_abs_P( 16.0/15.0, -4.0/15.0, 1.0/15.0 );
	      for (i=0; i<nsh; i++) {
		dd->ne[i] += CPC[i];
	      }
	      
#ifdef DEBUG
	      if (FULL_DEBUG || pp->dam == TEST_SIRE) {
		sprintf( msg, "mgs %d ne after P (HS) prog absorb", (int) (dd-ped) );
		flpr( stdout, msg, dd->ne, nsh );
	      }
#endif
	      
	    }
	    // derive contribution to sire
	    if (pp->sire) {
	    // absorb progeny into sire, prep for progeny absorption
	      mtedc_abs_P( 4.0/3.0, -2.0/3.0, 1.0/3.0 );
	      for (i=0; i<nsh; i++) {
		ss->ne[i] += CPC[i];
	      }
#ifdef DEBUG
	      if (FULL_DEBUG || pp->sire == TEST_SIRE) {
		sprintf( msg, "sire %d ne after (HS) prog absorb", (int) (ss-ped) );
		flpr( stdout, msg, ss->ne, nsh );
	      }
#endif
#ifdef DEBUG_CUSTOM
	      if (pp->sire == TEST_SIRE && ss->npe) {
		if (ss->npe) {
		  sprintf( msg, "Sire %d npe", TEST_SIRE );
		  flpr( stdout, msg, ss->npe, nsh );
		}
	      }
#endif
	    }

	  } // if (x)
	} // if (pp->sire || pp->dam)
      } // for (p=nanml; p>=1; p--) 
    }  // if (RELIABILITIES)

  }else if (nmaternal) {
      
    // Animal Model
    for (p=nanml; p>=1; p--) {
      pp = ped + parents_first[p];
      if (!(p%100000)) {
	fprintf( stdout, "%9d) %9d%9d%9d\n", p, (int) (pp-ped), pp->sire, pp->dam );
      }
#ifdef DEBUG
      if (FULL_DEBUG || 
	  (TEST_AN && (pp-ped) == TEST_AN) ||
	  (TEST_SIRE && (pp->sire) == TEST_SIRE) ||
	  (TEST_DAM && (pp->dam) == TEST_DAM)) {
	fprintf( stdout, "%9d) %9d%9d%9d [%d]\n", p, (int) (pp-ped), pp->sire, pp->dam, non_sire(pp) );
	fprintf( stdout, "Absorbing progeny %d\n", (int) (pp-ped) );
	sprintf( msg, " %d) ne", (int) (pp-ped) );
	flpr( stdout, msg, pp->ne, nsh );
	sprintf( msg, " %d) me (MxD)", (int) (pp-ped) );
	flpr2( stdout, msg, pp->me, ntraits, ntraits, ntraits );
      }
#endif
      
#ifdef NEW_ET_ADJ
      // ET calf and recipients are order chronologically...
      // absorb all calves into recips first, then all recips into their maternal progeny
      if (pp->matdam && pp->matdam != pp->dam) {
	// absorb ET calf into recipient cow
	dd = ped + pp->matdam;
	mtedc_abs_PR();
	for (i=0; i<nsh; i++) dd->ne[i] -= CPC[i];
	if (dd->npe) for (i=0; i<nsh; i++) dd->npe[i] -= CPC[i];
      }
#endif
      if ( (pp->sire || pp->dam) ) {
	ss = ped + pp->sire;
	dd = ped + pp->dam;
	// progeny contribution is zero unless ne > 0
	z = 4.0 / (2.0 + !pp->sire + !pp->dam);
	zAA = z;
	zAS = zAD = -.5 * z;
	zSS = zDD = zSD = 0.25 * z;
	x = 0.0;
	i = 0;
	while( !x && i<neffects ) {
	  x += pp->ne[irc(i,i)];
	  i++;
	}

	if (x) {
	    
	  // ne > 0 ... absorb the progeny
	  x = 0.0;
	  i = 0;
	  while( !x && i<nmaternal ) {
	    x += pp->me[i*ntraits+i];
	    i++;
	  }
	  if (x) {
	    if (pp->dam) {
	      // progeny X dam co-incidence => contribution to dam|mgs and sire differ
	      
	      if (!meyer_absorb) {
		// absorb progeny into dam, prep for mate absorption
		mtedc_abs_P2D( zAA, zAD, zDD );  // still need to absorb sire into dam
		for (i=0; i<nsh; i++) dd->npe[i] += CPC[i];
#ifdef DEBUG
		if (FULL_DEBUG || pp->dam == TEST_DAM) {
		  sprintf( msg, "dam %d npe after P2D (FS) prog absorb", pp->dam );
		  flpr( stdout, msg, dd->npe, nsh );
		}
#endif
	      }
	      
		
	      /*
#ifdef DEBUG
		if (FULL_DEBUG || (dd-ped) == TEST_SIRE) {
		  fprintf( stdout, "dam %d ne before PM (HS) prog absorb:\n", (int) (dd-ped) );
		  for (i=0; i<nsols; i++) {
		    for (j=0; j<=i; j++) fprintf( stdout, "%7.4f", dd->ne[irc(i,j)] );
		    fprintf( stdout, "\n" );
		  }
		}
#endif
	      */

	      // absorb progeny into dam, prep for progeny absorption
	      // assume sire unknown (zero contribution to dam/mgs)
	      mtedc_abs_PM( 4.0/3.0, -2.0/3.0, 1.0/3.0 );
	      for (i=0; i<nsh; i++) {
		dd->ne[i] += CPC[i];
	      }
	      
#ifdef DEBUG
	      if (FULL_DEBUG || pp->dam == TEST_DAM) {
		sprintf( msg, "dam %d ne after PM (HS) prog absorb", pp->dam );
		flpr( stdout, msg, dd->ne, nsh );
	      }
#endif
	      
	    }
	    // derive contribution to sire
	    if (non_sire( pp ) && pp->sire) {
	      if (!meyer_absorb) {
		// absorb progeny into sire, prep for mate absorption
		mtedc_abs_P2S( zAA, zAS, zSS );
		if (!meyer_absorb) {
		  for (i=0; i<nsh; i++) ss->npe[i] += CPC[i];
		}
	      }
	      
	      // absorb progeny into sire, prep for progeny absorption
	      mtedc_abs_P( 4.0/3.0, -2.0/3.0, 1.0/3.0 );
	      for (i=0; i<nsh; i++) {
		ss->ne[i] += CPC[i];
	      }
#ifdef DEBUG
	      if (FULL_DEBUG || pp->sire == TEST_SIRE) {
		sprintf( msg, "sire %d ne after (HS) prog absorb", (int) (ss-ped) );
		flpr( stdout, msg, ss->ne, nsh );
	      }
#endif
#ifdef DEBUG_CUSTOM
	      if (pp->sire == TEST_SIRE && ss->npe) {
		if (ss->npe) {
		  sprintf( msg, "Sire %d npe", TEST_SIRE );
		  flpr( stdout, msg, ss->npe, nsh );
		}
	      }
#endif
	    }
	  }else if (non_sire(pp)) {
	    // derive equivalent contributions to dam and sire
	    if (!meyer_absorb) {
	      mtedc_abs_P2S( zAA, zAS, zSS );
	      if (pp->dam) {
		// absorb progeny into dam, prep for mate absorption
		for (i=0; i<nsh; i++) dd->npe[i] += CPC[i];
	      }
	      if (pp->sire) {
		// absorb progeny into sire, prep for mate absorption
		for (i=0; i<nsh; i++) {
		  ss->npe[i] += CPC[i];
		}
#ifdef DEBUG
		if (FULL_DEBUG || pp->sire == TEST_SIRE) {
		  sprintf( msg, "Sire %d npe", (int) (ss-ped) );
		  flpr( stdout, msg, ss->npe, nsh );
		}
#endif
	      }

	    }
	    mtedc_abs_P( 4.0/3.0, -2.0/3.0, 1.0/3.0 );
	    if (pp->sire) {
	      // absorb progeny into sire, prep for progeny absorption
	      for (i=0; i<nsh; i++) {
		ss->ne[i] += CPC[i];
	      }
#ifdef DEBUG_CUSTOM
	      if (pp->sire == TEST_SIRE || (pp->dam && pp->dam == TEST_DAM)) {
		if (ss->npe) {
		  sprintf( msg, "Sire %d npe", TEST_SIRE );
		  flpr( stdout, msg, ss->npe, nsh );
#ifdef RELIABILITY_MODULE
		  flpr_rel( pp->npe, "Rel_npe(P)" );
		  flpr_rel( ss->npe, "Rel_npe(S)" );
		  if (pp->dam) flpr_rel( dd->npe, "Rel_npe(D)" );
#endif
		}
		sprintf( msg, "Sire %d ne", TEST_SIRE );
		flpr( stdout, msg, ss->ne, nsh );
#ifdef RELIABILITY_MODULE
		flpr_rel( pp->ne, "Rel_ne(P)" );
		flpr_rel( ss->ne, "Rel_ne(S)" );
		if (pp->dam) flpr_rel( dd->ne, "Rel_ne(D)" );
#endif
	      }
#endif
	    }
	    if (pp->dam) {
	      if (sire_model) mtedc_abs_P( 16.0/15.0, -4.0/15.0, 1.0/15.0 );
	      // absorb progeny into dam, prep for progeny absorption
	      for (i=0; i<nsh; i++) {
		dd->ne[i] += CPC[i];
	      }
	    }
	  }
	} // if (x)
      } // if (pp->sire || pp->dam)
    } // for (p=nanml; p>=1; p--) 
      
  }else {
      
    // Animal Model
    // progeny absorption is the same for sire and dam
    for (p=nanml; p>=1; p--) {
      pp = ped + parents_first[p];
      if (!(p%100000)) {
	fprintf( stdout, "%9d) %9d%9d%9d\n", p, (int) (pp-ped), pp->sire, pp->dam );
      }
#ifdef DEBUG
	if (FULL_DEBUG || 
	    (TEST_AN && (pp-ped) == TEST_AN) ||
	    (TEST_SIRE && (pp->sire) == TEST_SIRE) ||
	    (TEST_DAM && (pp->dam) == TEST_DAM)) {
	  fprintf( stdout, "%9d) %9d%9d%9d [%d]\n", p, (int) (pp-ped), pp->sire, pp->dam, non_sire(pp) );
	  fprintf( stdout, "Absorbing progeny %d\n", (int) (pp-ped) );
	  sprintf( msg, " %d) ne", (int) (pp-ped) );
	  flpr( stdout, msg, pp->ne, nsh );
	}
#endif
      if ( non_sire( pp ) && (pp->sire || pp->dam) ) {
	ss = ped + pp->sire;
	dd = ped + pp->dam;
	// progeny contribution is zero unless ne > 0
	z = 4.0 / (2.0 + !pp->sire + !pp->dam);
	zAA = z;
	zAS = zAD = -.5 * z;
	zSS = zDD = zSD = 0.25 * z;
	x = 0.0;
	i = 0;
	while( !x && i<neffects ) {
	  x += pp->ne[irc(i,i)];
	  i++;
	}
	
	if (x) {
	  // ne > 0
	  // derive equivalent contributions to dam and sire
	  if (!meyer_absorb) {
	    mtedc_abs_P2S( zAA, zAS, zSS );
	    if (pp->dam) {
	      // absorb progney into dam, prep for mate absorption
	      for (i=0; i<nsh; i++) {
		dd->npe[i] += CPC[i];
	      }
	    }
	    if (pp->sire) {
	      // absorb progeny into sire, prep for mate absorption
	      for (i=0; i<nsh; i++) {
		ss->npe[i] += CPC[i];
	      }
	    }
	  }
	  mtedc_abs_P( 4.0/3.0, -2.0/3.0, 1.0/3.0 );
	  if (pp->dam) {
	    // absorb progney into dam, prep for progeny absorption
	    for (i=0; i<nsh; i++) {
	      dd->ne[i] += CPC[i];
	    }
#ifdef DEBUG_CUSTOM
	    if (pp->dam == TEST_DAM) {
	      sprintf( msg, "Absorbing prog %d ne", (int) (pp-ped) );
	      flpr( stdout, msg, pp->ne, nsh );
	      sprintf( msg, "Absorbing prog %d npe", (int) (pp-ped) );
	      flpr( stdout, msg, pp->npe, nsh );
	      sprintf( msg, "Dam %d npe", TEST_DAM );
	      flpr( stdout, msg, dd->npe, nsh );
	    }
#endif
	  }
	  if (pp->sire) {
	    // absorb progeny into sire, prep for progeny absorption
	    for (i=0; i<nsh; i++) {
	      ss->ne[i] += CPC[i];
	    }
#ifdef DEBUG_CUSTOM
	    if (pp->sire == TEST_SIRE) {
	      sprintf( msg, "Absorbing prog %d ne", (int) (pp-ped) );
	      flpr( stdout, msg, pp->ne, nsh );
	      sprintf( msg, "Absorbing prog %d npe", (int) (pp-ped) );
	      flpr( stdout, msg, pp->npe, nsh );
	      sprintf( msg, "Sire %d npe", TEST_SIRE );
	      flpr( stdout, msg, ss->npe, nsh );
	    }
#endif
	    /*
	      pr( stdout, "C", C, nsols*nsols );
	      pr( stdout, "P", P, nsh );
	      printf( "Psire %d += %f (z=%f) = %f\n", ss-ped, CPC[0], z, ss->ne[0] );
	    */
	  }
	}
      }
    }
      
  }

#ifdef DEBUG
  if (FULL_DEBUG) {
    fprintf( stdout, "After Absorb progeny\n" );
    for_debug_list {
      if (i<=nanml && ped[i].ne) {
	sprintf( msg, "ped[%d].ne", i );
	flpr( stdout, msg, ped[i].ne, nsh );
      }
    }
  }else {
    if ((i=TEST_AN)) {
      fprintf( stdout, "After Absorb progeny\n" );
      sprintf( msg, " %d) ne", i );
      flpr( stdout, msg, ped[i].ne, nsh );
    }
    if ((i=TEST_SIRE)) {
      fprintf( stdout, "After Absorb progeny\n" );
      sprintf( msg, " %d) ne", i );
      flpr( stdout, msg, ped[i].ne, nsh );
    }
    if ((i=TEST_DAM)) {
      fprintf( stdout, "After Absorb progeny\n" );
      sprintf( msg, " %d) ne", i );
      flpr( stdout, msg, ped[i].ne, nsh );
    }
  }
#endif

  return;
}

//**********************************************************************
void mtedc_abs_RCOMBINATION( REAL *v, int nnn, int invert_ne ) {
  // convert Ne by trait to sums of R^ sub-matrices
  //   => makes assumptions about the underlying record combinations
  int i, j, ij, nh, n2, incidence;
  REAL min, z;

  nh = nnn*(nnn+1)/2;
  n2 = (int) pow( 2.0, (double) nnn );

  // move diagonals of ne to beginning of array to simplify following logic
  for (i=1; i<nnn; i++) v[i] = v[irc(i,i)];

  if (invert_ne) {
    // invert 1/ne (from ST-EDC section) to get ne
    for (i=0; i<nnn; i++) {
      if (v[i]) v[i] = 1.0 / v[i];
    }
  }

  // deduce combinations from ne by trait
  ncomb = 0;
  incidence = 1;
  while( incidence ) {
    incidence = 0;
    min = 0;
    for (i=0; i<nnn; i++) {
      if ((z=v[i])) {
	if (!min || z<min) min=z;
      }
    }
    if (min) {
      // another trait combination is observed
      for (i=0; i<nnn; i++) {
	if (v[i] >= min) {
	  incidence |= (1<<i);
	  v[i] -= min;
	}
      }
      combination[ncomb] = min;
      combination_inc[ncomb] = incidence;
      ncomb++;
    }
  }
  // now sum combination * corresponding Rinv matrices into ne
  for (i=0; i<nh; i++) v[i] = 0.0;
  for (i=0; i<ncomb; i++) {
    incidence = combination_inc[i];
    z = combination[i];
    for (j=0; j<Rinv_n[incidence] && 
	   (ij=Rinv_ij[incidence][j]) < nh; j++) {
      v[ij] += z * Rinv[incidence][j];
    }
  }
  return;
}

//**********************************************************************
void mtedc_abs_RECIPS() {
  // absorb prog and recips into each other, then clear ne[DxM]
  // this steps removes all ET complications from the rest of the algorythm
  int i, j;
  printf( "absorb RECIPS\n" );
  // first absorb progeny
  for (pp=ped+1; pp<=ped+nanml; pp++) {
    if (pp->matdam && pp->matdam != pp->dam) {
      dd = ped + pp->matdam;
      mtedc_abs_PR();
      for (i=0; i<nsh; i++) dd->ne[i] -= CPC[i];
      if (dd->npe) for (i=0; i<nsh; i++) dd->npe[i] -= CPC[i];
    }
  }
  // now absorb recips
  for (pp=ped+1; pp<=ped+nanml; pp++) {
    if (pp->matdam && pp->matdam != pp->dam) {
      dd = ped + pp->matdam;
      // de-absorb this progeny
      for (i=0; i<nsh; i++) Pdam[i] = dd->ne[i];
      mtedc_abs_PR();
      for (i=0; i<nsh; i++) Pdam[i] += CPC[i];
      mtedc_abs_RP();
      for (i=0; i<nsh; i++) pp->ne[i] -= CPC[i];
      if (pp->npe) for (i=0; i<nsh; i++) pp->npe[i] -= CPC[i];
      // replace DxM with zero co-incidence for the genetic dam
      for (i=0; i<ntraits; i++) {
	for (j=0; j<nmaternal; j++) pp->me[i*ntraits+j] = 0.0;
      }
      // go back to dd->ne before progeny was de-absorbed
      for (i=0; i<nsh; i++) dd->ne[i] = Pdam[i];
    }
  }
  return;
}

//**********************************************************************
void mtedc_abs_RINV( int invert_ne ) {
  // Deduce frequency of all trait combinations for proper accumulations of R-inv
  // input: ne[] = Dir and DxM   me[] = Mat
  // output: ne[] = D'rD and M'rD (of dam)   me[] = M'rM (of animal)
  int i, j, ij;

  //#### combination ...
  for (pp=ped+1; pp<=ped+nanml; pp++) {

    /*
    if (!((pp-ped)%10000)) {
      printf( "  Rinv for Animal %d\n", (int) (pp-ped) );
      PRINTMEM;
    }
    */

    for (i=0; i<nth; i++) P[i] = pp->ne[i];
    mtedc_abs_RCOMBINATION( P, ntraits, invert_ne );
    for (i=0; i<nth; i++) pp->ne[i] = P[i];

    for (i=0; i<nmh; i++) P[i] = pp->me[i];
    mtedc_abs_RCOMBINATION( P, nmaternal, invert_ne );
    for (i=0; i<nmh; i++) pp->me[i] = P[i];

#ifdef DEBUG
    if ((i=pp-ped) == TEST_AN) {
      sprintf( msg, " %d) ne after abs_RCOMBINATION", i );
      flpr( stdout, msg, ped[i].ne, nsh );
      sprintf( msg, " %d) me after abs_RCOMBINATION", i );
      flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
    }
    if ((i=pp-ped) == TEST_SIRE) {
      sprintf( msg, " %d) ne after abs_RCOMBINATION", i );
      flpr( stdout, msg, ped[i].ne, nsh );
      sprintf( msg, " %d) me after abs_RCOMBINATION", i );
      flpr2( stdout, msg, ped[i].me, ntraits, ntraits, ntraits );
    }
#endif

    // extract diagonals from full-store into half-store order
    for (i=0; i<nmaternal; i++) P[irc(i,i)] = pp->ne[nth+i*nmaternal+i];
    mtedc_abs_RCOMBINATION( P, nmaternal, invert_ne );
    // expand half-store result to full-store ne(direct X maternal)
    ij = 0;
    for (i=0; i<nmaternal; i++) {
      for (j=0; j<=i; j++) {
	pp->ne[nth+i*nmaternal+j] = pp->ne[nth+j*nmaternal+i] = P[ij];
	ij++;
      }
    }

  }
  return;
}

//**********************************************************************
void mtedc_abs_RP() {
  //derive ET recip contribution to progeny (input ET-recip ne[]=D and prog me[]=DxM)
  int i, j, k, ij, ii, jj;
  REAL x, z;
  for (i=0; i<nsh; i++) P[i] = 0.0;
  for (i=0; i<nsols; i++) {
    for (j=0; j<nsols; j++) {
      CC(i,j) = 0.0;
    }
  }
  ij = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      P[ij] = G2[ij];
      ij++;
    }
  }
  for (i=0; i<nsh; i++) P[i] += Pdam[i];
  for (i=0; i<ntraits; i++) {
    for (j=0; j<nmaternal; j++) {
      jj = ntraits + j;
      if (!nGeq0 || ( (G[irc(i,i)] && G[irc(jj,jj)]) )) {
	CC(i,j+ntraits) += pp->me[i*ntraits+j];   // direct X maternal  (row = absorb target = prog)
      }
    }
  }
  if (PEvar) {
    for (ii=0; ii<nPEdir; ii++) {
      i = PEdir_trait[ii];
      for (j=0; j<nmaternal; j++) {
	CC(ii+neffects,j+ntraits) += pp->me[i*ntraits+j];   // PEdir X maternal  (row = absorb target = prog)
      }
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(ii+neffects,jj+neffects+nPEdir) += pp->me[i*ntraits+j];   // PEdir X PEmat  (row = absorb target = prog)
      }
    }
    for (i=0; i<ntraits; i++) {
      for (jj=0; jj<nPEmat; jj++) {
	j = PEmat_trait[jj];
	CC(i,jj+neffects+nPEdir) += pp->me[i*ntraits+j];   // direct X PEmat  (row = absorb target = prog)
      }
    }
  }
#ifdef DEBUG
  if ((pp-ped) == TEST_AN) {
    printf( "abs_RP\n" );
    printf( "  C:\n" );
    for (i=0; i<nsols; i++) {
      for (j=0; j<nsols; j++) printf( "%9.5f", CC(i,j) ); printf( "\n" );
    }
    pr( stdout, "P", P, nsh );
  }
#endif
#ifdef SLOW
  invert( P, nsols );
  f_CPC( 0, nsols, 0, nsols );
#else
  f2_CPC( 0, nsols, 0, nsols );
#endif
#ifdef DEBUG
  if ((pp-ped) == TEST_AN) {
    pr( stdout, "CPC", CPC, nsh );
  }
#endif
  return;
}

//**********************************************************************
int non_sire( PED *p ) {
  // 
  return (!sire_model && (RELIABILITIES || p->edc == NULL));
}

//**********************************************************************
void open_files( ) {
  // open input/output files and process the parameter file, options, etc.
  char suff[28];
  int i, j, n;
 
  strcpy( fnames[PARAMFILE], "EDCparam.dat" );
  if (param_file != NULL) strcpy( fnames[PARAMFILE], param_file );
  strcpy( fnames[PEDFILE], "EDCped.dat" );
  strcpy( fnames[DATAFILE], "EDCobs.dat" );
  strcpy( fnames[EDCFILE], "EDC.out" );
  strcpy( fnames[LOGFILE], "EDC.log" );
  strcpy( fnames[CHKFILE], "EDC.chk" );
  strcpy( fnames[SASFILE], "EDC.sas" );
  strcpy( fnames[TRUFILE], "EDC.true" );

  *fnames[PEDFILE] = *fnames[DATAFILE] = *fnames[EDCFILE] = *fnames[LOGFILE] =
    *fnames[CHKFILE] = *fnames[SASFILE] = *fnames[TRUFILE] = 0;

  sprintf( fnames[STDERR], "%s.stderr", param_file );
  sprintf( fnames[STDOUT], "%s.stdout", param_file );
  if (ignore_ETs) {
    strcpy( fnames[STDOUT]+strlen(fnames[STDOUT]), "_noET" );
    strcpy( fnames[STDERR]+strlen(fnames[STDERR]), "_noET" );
  }
  if (redirect_output) {
    //    printf( "redirecting screen messages to: [%s] and [%s]\n",
    //	    fnames[STDOUT], fnames[STDERR] );
    checkdir( fnames[STDERR] );
    sprintf( fname_stderr, "%s", fnames[STDERR] );
    freopen( fname_stderr, "w", stderr );
    setbuf( stderr, 0 );
    f[STDERR] = stderr;

    checkdir( fnames[STDOUT] );
    sprintf( fname_stdout, "%s", fnames[STDOUT] );
    freopen( fname_stdout, "w", stdout );
    setbuf( stdout, 0 );
    f[STDOUT] = stdout;
    fputs( cmdline, stdout );
  }

  fpar = f[PARAMFILE] = fop( fnames[PARAMFILE], "r" );
  if (fpar == NULL) {
    sprintf( msg, "error opening parameter file [%s]\n", fnames[PARAMFILE] );
    error( msg );
  }

  printf( "reading parameter file \"%s\"...\n", param_file );
  read_parameters();
  // Some Arrays are now allocated based on the number of traits and maximum effects possible.
  // The actual number of effects will not be known until after 'summarize_parameters' below.

  if (redirect_stdout && !redirect_output) {
    fputs( cmdline, stdout );
    RAMmonitor = RAMmonitor_0;
    update_allocated(0);
  }

  if (call_edc_pgm) {
    sprintf( EDC_method, "Custom EDC program was used (e.g. Zengting's MT-EDC Program)\n" );
  }else if (NDAU_NE) {
    sprintf( EDC_method, "EDC set equal to number of daughters\n" );
  }else if (ITB_NE) {
    sprintf( EDC_method, "EDC based on Interbull 2000 method (mimeo Freddy Fikse, July)\n" );
  }else if (MEYER_NE) {
    sprintf( EDC_method, "EDC based on Interbull 2000 method (mimeo Freddy Fikse, July)\n" );
    sprintf( EDC_method+strlen( EDC_method), 
	     "  with simpler adjustment for CG estimation (K. Meyer approach)\n" );
  }else if (PGS1_NE) {
    sprintf( EDC_method, "EDC based on Sullivan et al, 2004 (ITB Bulletin 32:53-58)\n" );
  }else if (PGS2_NE) {
    sprintf( EDC_method, "EDC test2 after Sullivan et al, 2004 (ITB Bulletin 32:53-58)\n" );
  }else if (PGS3_NE) {
    sprintf( EDC_method, "EDC test3 after Sullivan et al, 2004 (ITB Bulletin 32:53-58)\n" );
  }else if (PGS4_NE) {
    sprintf( EDC_method, "EDC based on Sullivan et al, 2006 (ITB Bulletin 35:112-116)\n" );
    sprintf( EDC_method+strlen( EDC_method), 
	     "  ERRATA: Step 5) Do not absorb sires into their parents.\n" );
    sprintf( EDC_method+strlen( EDC_method), 
	     "          This correction avoids double counting in MACE.\n" );
    sprintf( EDC_method+strlen( EDC_method),
	     "  UPDATE: Step 1) extended from single-trait to multi-trait, thus\n" );
    sprintf( EDC_method+strlen( EDC_method),
	     "          Step 2) deleted as it is no longer needed.\n" );
  }else if (RELIABILITIES) {
    sprintf( EDC_method, "Approximating reliabilities for all animals (instead of EDCs for sires)\n" );
  }else {
    sprintf( EDC_method, "EDC methodology NOT SPECIFIED !!!\n" );
  }

  if (call_edc_pgm) {
    sprintf( suff, "_%s", edc_suf );
  }else if (default_WF) {
    strcpy( suff, "" );
  }else if (NDAU_NE) {
    strcpy( suff, "_ND" );
  }else if (ITB_NE) {
    strcpy( suff, "_ITB" );
  }else if (MEYER_NE) {
    strcpy( suff, "_Ne" );
  }else if (PGS1_NE) {
    strcpy( suff, "_PGS1" );
  }else if (PGS2_NE) {
    strcpy( suff, "_PGS2" );
  }else if (PGS3_NE) {
    strcpy( suff, "_PGS3" );
  }else if (PGS4_NE) {
    //strcpy( suff, "_PGS7" );  // went up to 7 with research for ITB2006 meeting
    //strcpy( suff, "_PGS4" );    // going back to 4 for production release of software
    if (advanced_mate_absorb) {
      strcpy( suff, "_PGS5" );
    }else {
      strcpy( suff, "_PGS4" );
    }
  }else if (RELIABILITIES) {
    if (output_accuracies) {
      if (meyer_absorb) {
	strcpy( suff, "_ACCm" );
      }else if (advanced_mate_absorb) {
	strcpy( suff, "_ACC5" );
      }else {
	strcpy( suff, "_ACC4" );
      }
    }else {
      if (meyer_absorb) {
	strcpy( suff, "_RELm" );
      }else if (advanced_mate_absorb) {
	strcpy( suff, "_REL5" );
      }else {
	strcpy( suff, "_REL4" );
      }
    }
  }else {
    strcpy( suff, "" );
  }

  if (check_SI) {
    strcpy( suff+strlen(suff), "_SI" );
    if (true_suff[0]) strcpy( true_suff+strlen(true_suff), "_SI" );
  }
  if (bigCG) strcpy( suff+strlen(suff), "_1CG" );
#ifdef SLOW
  strcpy( suff+strlen(suff), "_SLOW" );
#endif

  if (redirect_output || redirect_stderr) {
    checkdir( fnames[STDERR] );
    sprintf( fname, "%s%s", fnames[STDERR], suff );
    sprintf( msg, "rm -f %s && mv -f %s %s", fname, fnames[STDERR], fname );
    fclose( stderr );
    system( msg );
    strcpy( fnames[STDERR], fname );
    freopen( fnames[STDERR], "a", stderr );
    setbuf( stderr, 0 );
    f[STDERR] = stderr;
  }
  if (redirect_output || redirect_stdout) {
    checkdir( fnames[STDOUT] );
    sprintf( fname, "%s%s", fnames[STDOUT], suff );
    sprintf( msg, "rm -f %s && mv -f %s %s", fname, fnames[STDOUT], fname );
    fclose( stdout );
    system( msg );
    strcpy( fnames[STDOUT], fname );
    freopen( fnames[STDOUT], "a", stdout );
    setbuf( stdout, 0 );
    f[STDOUT] = stdout;
  }

  if (ignore_ETs) {
    if (*fnames[EDCFILE]) strcpy( fnames[EDCFILE]+strlen(fnames[EDCFILE]), "_noET" );
    if (*fnames[LOGFILE]) strcpy( fnames[LOGFILE]+strlen(fnames[LOGFILE]), "_noET" );
    if (*fnames[CHKFILE]) strcpy( fnames[CHKFILE]+strlen(fnames[CHKFILE]), "_noET" );
    if (*fnames[SASFILE]) strcpy( fnames[SASFILE]+strlen(fnames[SASFILE]), "_noET" );
    if (*fnames[TRUFILE]) strcpy( fnames[TRUFILE]+strlen(fnames[TRUFILE]), "_noET" );
  }
  if (runcheck &&true_suff[0]) {
    sprintf( fname_true_rel, "%s%s", fnames[SASFILE], true_suff );
  }
  //##########
  //  strcpy( fnames[SASFILE]+strlen(fnames[SASFILE]), "_tmp" );
  //##########
  if (suff[0]) {
    if (*fnames[EDCFILE]) strcpy( fnames[EDCFILE]+strlen(fnames[EDCFILE]), suff );
    if (*fnames[LOGFILE]) strcpy( fnames[LOGFILE]+strlen(fnames[LOGFILE]), suff );
    if (*fnames[CHKFILE]) strcpy( fnames[CHKFILE]+strlen(fnames[CHKFILE]), suff );
    if (*fnames[SASFILE]) strcpy( fnames[SASFILE]+strlen(fnames[SASFILE]), suff );
    if (*fnames[TRUFILE]) strcpy( fnames[TRUFILE]+strlen(fnames[TRUFILE]), suff );
  }
  f[PEDFILE] = fped = fop( fnames[PEDFILE], "r" );
  f[DATAFILE] = frec = fop( fnames[DATAFILE], "r" );
  f[LOGFILE] = flog = fop( fnames[LOGFILE], "w" );
  if (flog) setbuf( flog, 0 );
  if (runcheck) {
    f[CHKFILE] = fcheck = fop( fnames[CHKFILE], "w" );
    if (fcheck) setbuf( fcheck, 0 );
    f[SASFILE] = fsas = fop( fnames[SASFILE], "w" );
    if (fsas) setbuf( fsas, 0 );
    f[TRUFILE] = ftru = fop( fnames[TRUFILE], "w" );
    if (ftru) setbuf( ftru, 0 );
  }
  f[EDCFILE] = fout = fop( fnames[EDCFILE], "w" );
  //  setbuf( f[EDCFILE], NULL );
  if (RELIABILITIES) frel = f[EDCFILE];

  nflds[PEDFILE] = 3;

  summarize_parameters();
  // Adjust nsols now that we know the exact # of PE effects in the model
  nsols = neffects + nPEdir + nPEmat;
  nsh = (nsols*(nsols+1))/2;

  return;
}

//**********************************************************************
void output_edcs() {
  // write to ouptut file and accumulate info. for log file
  int i, j, k, t, neg_EDCs, neg_count, sire_count;
  REAL x;
  if (output_integer) {
    sprintf( fmt, "%%7.0f" );
  }else {
    sprintf( fmt, "%%12.4f" );
  }
  neg_count = sire_count = 0;
  for (pp=ped+1; pp<=ped+nanml; pp++) {
    if (pp->edc != NULL) {
      sire_count++;
      neg_EDCs = 0;
      k = 0;
      for (j=0; j<nEDC; j++) k |= (pp->np[j]>0);
      k = 1;
      if (k) {
        fprintf( fout, "%9d", (int) (pp-ped) );
#ifdef DEBUG_CUSTOM
	if ((pp-ped) == TEST_SIRE) {
	  printf( "%9d edc: ", (int) (pp-ped) );
	  for (i=0; i<nEDC; i++) printf( "%7.1f", ped[TEST_SIRE].edc[i] );
	  printf( "\n" );
	  printf( "%9d  np: ", (int) (pp-ped) );
	  for (i=0; i<nEDC; i++) printf( "%7.0f", pp->np[i] );
	  printf( "\n" );
	}
#endif
#define MAXPRT 999999
	for (t=0; t<nEDC; t++) {
	  if (pp->edc[t] < 0) {
	    if (pp->edc[t] <= -0.5) {
	      neg_EDCs = 1;
	    }else {
	      pp->edc[t] = 0;
	    }
	  }
	  if (delimited_output) fprintf( fout, "," );
	  if (pp->edc[t] > MAXPRT) {
	    fprintf( fout, fmt, MAXPRT );
	  }else {
	    fprintf( fout, fmt, pp->edc[t] );
	  }
	}
	for (t=0; t<nEDC; t++) {
	  if (delimited_output) fprintf( fout, "," );
	  if (pp->np[t] > MAXPRT) {
	    fprintf( fout, " %6d", MAXPRT );
	  }else {
	    fprintf( fout, " %6.0f", pp->np[t] );
	  }
	}
	if (count_grandprogeny) {
	  for (t=0; t<nEDC; t++) {
	    if (delimited_output) fprintf( fout, "," );
	    if (!pp->npgp) {
	      fprintf( fout, " %6d", 0 );
	    }else {
	      if (pp->npgp[t] > MAXPRT) {
		fprintf( fout, " %6d", MAXPRT );
	      }else {
		fprintf( fout, " %6.0f", pp->npgp[t] );
	      }
	    }
	  }
	  for (t=0; t<nEDC; t++) {
	    if (delimited_output) fprintf( fout, "," );
	    if (!pp->nmgp) {
	      fprintf( fout, " %6d", 0 );
	    }else {
	      if (pp->nmgp[t] > MAXPRT) {
		fprintf( fout, " %6d", MAXPRT );
	      }else {
		fprintf( fout, " %6.0f", pp->nmgp[t] );
	      }
	    }
	  }
	}
	if (edc_reliabilities) {
	  for (t=0; t<nEDC; t++) {
	    if (delimited_output) fprintf( fout, "," );
	    if (output_integer) {
	      fprintf( fout, fmt, (100.0 * pp->edc[t] / (pp->edc[t] + alpha[t])) );
	    }else {
	      fprintf( fout, fmt, (  1.0 * pp->edc[t] / (pp->edc[t] + alpha[t])) );
	    }
	  }
	}
        fprintf( fout, "\n" );

	if (neg_EDCs) {
	  if (neg_count < 100) {
	    fprintf( flog, "%9d", (int) (pp-ped) );
	    for (t=0; t<nEDC; t++) {
	      if (delimited_output) fprintf( flog, "," );
	      if (pp->edc[t] > MAXPRT) {
		fprintf( flog, fmt, MAXPRT );
	      }else {
		fprintf( flog, fmt, pp->edc[t] );
	      }
	    }
	    for (t=0; t<nEDC; t++) {
	      if (delimited_output) fprintf( flog, "," );
	      if (pp->np[t] > MAXPRT) {
		fprintf( flog, " %6d", MAXPRT );
	      }else {
		fprintf( flog, " %6.0f", pp->np[t] );
	      }
	    }
	    if (count_grandprogeny) {
	      for (t=0; t<nEDC; t++) {
		if (delimited_output) fprintf( flog, "," );
		if (!pp->npgp) {
		  fprintf( flog, " %6d", 0 );
		}else {
		  if (pp->npgp[t] > MAXPRT) {
		    fprintf( flog, " %6d", MAXPRT );
		  }else {
		    fprintf( flog, " %6.0f", pp->npgp[t] );
		  }
		}
	      }
	      for (t=0; t<nEDC; t++) {
		if (delimited_output) fprintf( flog, "," );
		if (!pp->nmgp) {
		  fprintf( flog, " %6d", 0 );
		}else {
		  if (pp->nmgp[t] > MAXPRT) {
		    fprintf( flog, " %6d", MAXPRT );
		  }else {
		    fprintf( flog, " %6.0f", pp->nmgp[t] );
		  }
		}
	      }
	    }
	    if (edc_reliabilities) {
	      for (t=0; t<nEDC; t++) {
		if (delimited_output) fprintf( flog, "," );
		if (output_integer) {
		  fprintf( flog, fmt, (100.0 * pp->edc[t] / (pp->edc[t] + alpha[t])) );
		}else {
		  fprintf( flog, fmt, (  1.0 * pp->edc[t] / (pp->edc[t] + alpha[t])) );
		}
	      }
	    }
	    fprintf( flog, "\n" );
	  }
	  neg_count++;
	}
	
	for (t=0; t<nEDC; t++) {
	  if (pp->np[t]) {
	    accum_mstat( means_np[0]+t, pp->edc[t] );
	    accum_mstat( means_np[1]+t, pp->np[t] );
	    x = pp->edc[t] - pp->np[t];
	    accum_mstat( means_np[2]+t, x );
	    if (x<0) x = -x;
	    accum_mstat( means_np[3]+t, x );
	    accum_mstat( means_np[4]+t, pp->edc[t] / pp->np[t] );
	  }
	  if (pp->edc[t] > 0) {
	    accum_mstat( means_edc[0]+t, pp->edc[t] );
	    accum_mstat( means_edc[1]+t, pp->np[t] );
	    x = pp->edc[t] - pp->np[t];
	    accum_mstat( means_edc[2]+t, x );
	    if (x<0) x = -x;
	    accum_mstat( means_edc[3]+t, x );
	    if (pp->np[t] > 0) {
	      accum_mstat( means[0]+t, pp->edc[t] );
	      accum_mstat( means[1]+t, pp->np[t] );
	      x = pp->edc[t] - pp->np[t];
	      accum_mstat( means[2]+t, x );
	      if (x<0) x = -x;
	      accum_mstat( means[3]+t, x );
	      accum_mstat( means[4]+t, pp->edc[t] / pp->np[t] );
	    }
	  }
	}
      }
    }
#ifdef DEBUG_CUSTOM
    if (!((pp-ped)%100000)) {
      printf( "%9d np: ", (int) (pp-ped) );
      for (i=0; i<nEDC; i++) printf( "%7.0f", ped[TEST_SIRE].np[i] );
      printf( "\n" );
    }
#endif
  }
#ifdef DEBUG
  if (FULL_DEBUG) {
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      if (pp->edc != NULL) {
	fprintf( flog, "%5d: EDCs ", (int) (pp-ped) );
	for (j=0; j<nEDC; j++) {
	  fprintf( flog, "%12.4f", pp->edc[j] );
	}
	fprintf( flog, "\n" );
	fprintf( flog, "%5d: RELs ", (int) (pp-ped) );
	for (j=0; j<nEDC; j++) {
	  fprintf( flog, "%12.4f", (pp->edc[j] / (pp->edc[j] + alpha[j])) );
	}
	fprintf( flog, "\n" );
      }
    }
  }
#endif
  //  fclose( fout ); fout = 0;
  if (neg_count) {
    warning( "\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n" );
    sprintf( msg, "*** Negative EDCs were computed for %d of %d sires (%.1f%%).\n",
	     neg_count, sire_count, ( (100.0 * neg_count) / sire_count ) );
    warning( msg );
    warning( "***    This is highly unexpected. Please contact CDN/Lactanet.\n" );
    warning( "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n" );
#ifndef DEBUG
    //    error( "" );
#endif
  }
 return;
}

//**********************************************************************
void output_rels() {
  // write to ouptut file and accumulate info. for log file
  char *FMTd, *FMTf;
  int i, neg_RELs, neg_count, an_count;
  REAL x;
  //setbuf( frel, NULL );
  if (delimited_output) {
    FMTd="%d";
    FMTf="%.5f";
  }else {
    FMTd="%9d";
    FMTf="%9.5f";
  }
  neg_count = an_count = 0;
  for (pp=ped+1; pp<=ped+nanml; pp++) {
    an_count++;
    neg_RELs = 0;
    fprintf( frel, FMTd, (int) (pp-ped) );
    for (i=0; i<nsh; i++) P[i] = G2[i] + pp->npe[i];
    invert( P, nsols );
    for (i=0; i<neffects; i++) {
      if (delimited_output) fprintf( frel, "," );
      if (G[irc(i,i)] > 0) {
	x = 1.0 - P[irc(i,i)] / G[irc(i,i)];
      }else {
	x = 0.0;
      }
      if (x < 0) {
	if (x <= -0.005) {
	  neg_RELs = 1;
	}else {
	  x = 0.0;
	}
      }
      if (output_accuracies) x = sqrt(x);
      fprintf( frel, FMTf, x );
    }
    for (i=0; i<nindex; i++) {
      if (delimited_output) fprintf( frel, "," );
      x = 1.0 - kbk( weights[i], P, neffects ) / kbk( weights[i], G, neffects );
      if (x < 0) {
	if (x <= -0.005) {
	  neg_RELs = 1;
	}else {
	  x = 0.0;
	}
      }
      if (output_accuracies) x = sqrt(x);
      fprintf( frel, FMTf, x );
    }
    fprintf( frel, "\n" );
    
    if (neg_RELs) {
      if (neg_count < 100) {
	fprintf( flog, FMTd, (int) (pp-ped) );
	for (i=0; i<neffects; i++) {
	  if (delimited_output) fprintf( flog, "," );
	  if (G[irc(i,i)] > 0) {
	    x = 1.0 - P[irc(i,i)] / G[irc(i,i)];
	  }else {
	    x = 0.0;
	  }
	  if (x < 0 && x > -0.005) x = 0.0;
	  fprintf( flog, FMTf, x );
	}
	for (i=0; i<nindex; i++) {
	  if (delimited_output) fprintf( flog, "," );
	  x = 1.0 - kbk( weights[i], P, neffects ) / kbk( weights[i], G, neffects );
	  if (x < 0 && x > -0.005) x = 0.0;
	  fprintf( flog, FMTf, x );
	}
	fprintf( flog, "\n" );
      }
      neg_count++;
    }
    
  }
  //  fclose( frel ); frel = 0;
  if (neg_count) {
    warning( "\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n" );
    sprintf( msg, "*** Negative Reliabilities were computed for %d of %d animals (%.1f%%).\n",
	     neg_count, an_count, ( (100.0 * neg_count) / an_count ) );
    warning( msg );
    warning( "***    This is highly unexpected. Please contact CDN/Lactanet.\n" );
    warning( "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n" );
#ifndef DEBUG
    //    error( "" );
#endif
  }
  return;
}

//**********************************************************************
void process_option( char *cc ) {
  // command-line options that can also be specified in the parameter file
  int i, j;
  if (!strcmp( cc, "ignore_h2_check" )) {
    ignore_h2_check = 1;
  }else if (!strcmp( cc, "ignore_MAXSIRES_check" )) {
    ignore_MAXSIRES_check = 1;
  }else if (!strcmp( cc, "ignore_PD_check" )) {
    ignore_PD_check = 1;
  }else if (!strcmp( cc, "ignore_WTS_check" )) {
    ignore_WTS_check = 1;
  }else if (!strcmp( cc, "ignore_RECWTS" )) {
    ignore_RECWTS = 1;      
  }else if (!strcmp( cc, "h2_within_CG" )) {
    h2_within_CG = 1;
  }else if (!strcmp( cc, "ignore_SIRExMGS" )) {
    ignore_SIRExMGS = 1;
  }else if (!strncmp( cc, "include_traits=", 15 )) {
    fprintf( stdout, "include_traits list [%s]\n", cc+15 );
    for (i=15,j=0; cc[i]; i++,j++) if (cc[i]=='+') line[j]=','; else line[j] = cc[i];
    line[j] = 0;
    split_line( 0, 0, 0 );
    n_include_traits = nvalues;
    include_traits = (int *) init( 0, sizeof(int), n_include_traits );
    for (i=0; i<n_include_traits; i++) {
      include_traits[i] = atoi( value[i+1] );
      if (!include_traits[i]) {
	fprintf( stderr, "(%s) this list must be comma-delimited positive integers (i.e. trait #s)\n",
		 cc );
	ex(9);
      }
      fprintf( stdout, " => including trait %d\n", include_traits[i] );
    }
  }else {
    sprintf( msg, "[%s] is not a valid OPTION\n", cc );
    error( msg );
  }
  return;
}

//**********************************************************************
void print_help( int code ) {
  fprintf( stdout, "\nUsage for crEDC_%s\n", version );
  fprintf( stdout, "--------------------------------------------\n\n" );
  fprintf( stdout, "crEDC [-h] [-d] [-o] [-A] [-R] [-S] [-p *] [w *]\n" );
  fprintf( stdout, "      -h : Usage (this message)\n" );
  fprintf( stdout, "      -d : Format output file as comma-delimited variable length\n" );
  fprintf( stdout, "        -H : Include a header row in the comma-delimited output file\n" );
  fprintf( stdout, "      -o : Redirect stdout and stderr to files\n" );
  fprintf( stdout, "      -A : Add EDC-based reliabilities to EDC output file\n" );
  fprintf( stdout, "      -G : Add grand-progeny record counts to EDC output file\n" );
  fprintf( stdout, "      -R : Output EDCs as Real (F12.4) instead of integer (I7)\n" );
  fprintf( stdout, "      -S : Selection index, ignore estimation of CG effects\n" );
  fprintf( stdout, "    -p * : Use parameter file * (REQUIRED OPTION)\n" );
  fprintf( stdout, "    -w * : Use weighting factor method *, options are:\n" );
  fprintf( stdout, "           ND => Number of Daughters with a record\n" );
  fprintf( stdout, "           Ne => Simple adjustment (1-1/n) for CG effect\n" );
  fprintf( stdout, "           ITB => As defined by Interbull, 2000\n" );
  fprintf( stdout, "           PGS1 => first attempt (Sousse 2004)\n" );
  fprintf( stdout, "           PGS4 => MT-absorption of progeny (Kuopio 2006)\n" );

#ifdef RELIABILITY_MODULE
  fprintf( stdout, "\nRELIABILITY module options (reliabilities instead of EDCs):\n" );
  fprintf( stdout, "    -w * : Use reliability method *, options are:\n" );
  fprintf( stdout, "           REL4 (consistent with -w PGS4 for EDCs)\n" );
  fprintf( stdout, "           RELm (uses Karin Meyer's general approximation strategy)\n" );
  fprintf( stdout, "                 => Xb adjustment accounts for # animals in CG\n" );
  fprintf( stdout, "                 => Absorptions ignores some mate information\n" );
  fprintf( stdout, "           ACC4  => ouput accuracies (sqrt(REL4)) instead of reliabilities\n" );
  fprintf( stdout, "           ACCm  => ouput accuracies (sqrt(RELm)) instead of reliabilities\n" );
  fprintf( stdout, "       REL4 is used by default if -w is not specified\n" );
#else
  fprintf( stdout, "       PGS4 is used by default if -w is not specified\n" );
#endif

#ifdef IN_DEVELOPMENT
  fprintf( stdout, "\nAdditional options during development:\n" );
  fprintf( stdout, "      -z : Ignore non-zero residual correlations\n" );
  fprintf( stdout, "      -B : Assume 1 big CG\n" );
  fprintf( stdout, "      -C : Do not do the check\n" );
  fprintf( stdout, "      -D : Ignore R(o)_dam when computing EDCs\n" );
  fprintf( stdout, "      -E : Call EDC program specified in parameter file\n" );
  fprintf( stdout, "      -M : Evaluate strictly maternal for MGS (i.e. no direct)\n" );
  fprintf( stdout, "      -N : include records for natural progeny only, ignore ETs\n" );
  fprintf( stdout, "      -O * : specify OPTION *, same as in the parameter file\n" );
  fprintf( stdout, "      -X : ignore record weights\n" );
  fprintf( stdout, "    -w * : Use weighting factor method *, additional options:\n" );
  fprintf( stdout, "           PGS2 => first try targeting sire contribution to Ro\n" );
  fprintf( stdout, "           PGS3 => PGS2 modified for Jette's comments\n" );
  fprintf( stdout, "           REL5 => new approximation method being developed\n" );
#endif

  fprintf( stdout, "\n" );  

  printing_help = 1;
  ex( code );
}

//**********************************************************************
int qsort_dam( const void *a, const void *b ) {
  // needed for MATE absorption routines
  int i1, i2;
  i1 = (*((PED **)a))->sire;
  i2 = (*((PED **)b))->sire;
  return (i1 - i2);
}

//**********************************************************************
int qsort_record( const void *a, const void *b ) {
  // needed for CG absorption routines
  int sire_a, sire_b, mgs_a, mgs_b;
  sire_a = (*((RECORD **)a))->an;
  mgs_a = (*((RECORD **)a))->dam;
  sire_b = (*((RECORD **)b))->an;
  mgs_b = (*((RECORD **)b))->dam;
  return ((sire_a > sire_b) || ( (sire_a == sire_b)&&(mgs_a > mgs_b) ));
}

//**********************************************************************
int qsort_sire( const void *a, const void *b ) {
  // needed for MATE absorption routines
  int i1, i2;
  i1 = (*((PED **)a))->dam;
  i2 = (*((PED **)b))->dam;
  return (i1 - i2);
}

//**********************************************************************
void read_parameters() {
  // process all input parameters and allocate RAM for vectors and arrays
#define LABEL_LEN 9
  char tmp[BSZ];
  int i, j, k, ii, ll, ofs;
  int row[4], element[4], nelements[4], norder[4];
  fgets( line, BSZ, fpar );
  split_line( 3, fnames[PARAMFILE], 1 );
  ntraits = atoi( value[1] );
  nmaternal = atoi( value[2] );
  neffects = ntraits + nmaternal;
  neff2 = neffects * 2;
  nsols = (neffects + nmaternal) * 2;
  nobs = ntraits + 2 * nmaternal;
  nindex = atoi( value[3] );
  nEDC = nindex;
  nei = neffects + nindex;
  if (ntraits > (8*sizeof(int))-1) {
    sprintf( msg, "Too many traits: %d > Max %d\n", ntraits, (int) ((8*sizeof(int)) - 1) ) ;
    error( msg );
  }
  nt2p = 1 << ntraits;
  nth = (ntraits*(ntraits+1))/2;
  neh = (neffects*(neffects+1))/2;
  ne2h = (neff2*(neff2+1))/2;
  nsh = (nsols*(nsols+1))/2;
  noh = (nobs*(nobs+1))/2;
  nmh = (nmaternal*(nmaternal+1))/2;
  nm2 = nmaternal * nmaternal;
  nEDCh = (nEDC*(nEDC+1))/2;
  neih = (nei*(nei+1))/2;
  norder[0] = ntraits;
  norder[1] = neffects;
  norder[2] = neffects;
  norder[3] = ntraits;
  nelements[0] = nth;
  nelements[1] = neh;
  nelements[2] = neh;
  nelements[3] = nth;
  strcpy( matrix_label[0], "R" );
  strcpy( matrix_label[1], "G" );
  strcpy( matrix_label[2], "E" );
  strcpy( matrix_label[3], "CG" );
  for (i=0; i<4; i++) matrix[i] = (REAL *) zinit( matrix_label[i], 
						  sizeof(REAL), nelements[i] );
  R = matrix[0];
  G = matrix[1];
  E = matrix[2];
  CG = matrix[3];

  R2 = (REAL *) zinit( "R2", sizeof(REAL), neih );
  Rinv_tmp = (REAL *) init( "Rinv_tmp", sizeof(REAL), nth );
  Rinv_n = (int *) init( "Rinv_n", sizeof(int), nt2p );
  Rinv_ij = (int **) init( "Rinv_ij", sizeof(int *), nt2p );
  Rinv = (REAL **) init( "Rinv", sizeof(REAL **), nt2p );
  combination = (REAL *) init( "combination", sizeof(REAL), nt2p );
  combination_inc = (int *) init( "combination_inc", sizeof(int), nt2p );
  G2 = (REAL *) zinit( "G2", sizeof(REAL), neih + nsh );  // extra RAM avoids check max(nEDCh,nsh)
  S = (REAL *) zinit( "S", sizeof(REAL), neh );
  RS = (REAL *) zinit( "RS", sizeof(REAL), nth );
  GEDC = (REAL *) zinit( "GEDC", sizeof(REAL), nei );
  GEDC2 = (REAL *) zinit( "GEDC2", sizeof(REAL), nei );
  E2 = (REAL *) zinit( "E2", sizeof(REAL), neh );
  Geq0 = (int *) zinit( "Geq0", sizeof(int), neffects );
  PEdir = (int *) zinit( "PEdir", sizeof(int), ntraits );
  PEdir_trait = (int *) zinit( "PEdir_trait", sizeof(int), ntraits );
  if (nmaternal) {
    PEmat = (int *) zinit( "PEmat", sizeof(int), nmaternal );
    PEmat_trait = (int *) zinit( "PEmat_trait", sizeof(int), nmaternal );
  }
  vary = (REAL *) zinit( "vary", sizeof(REAL), nth );
  varg = (REAL *) zinit( "varg", sizeof(REAL), nth );
  P = (REAL *) zinit( "P", sizeof(REAL), noh + nsh );
  P2 = (REAL *) zinit( "P2", sizeof(REAL), noh + nsh );
  Pdam = (REAL *) zinit( "Pdam", sizeof(REAL), noh + nsh );
  Pprog = (REAL *) zinit( "Pprog", sizeof(REAL), noh + nsh );
  Psire = (REAL *) zinit( "Psire", sizeof(REAL), noh + nsh );
  B = (REAL *) zinit( "B", sizeof(REAL), (nobs+nsols)*neff2 );
  C = (REAL *) zinit( "C", sizeof(REAL), (nobs+nsols)*neff2 );
  Cdm = (REAL *) zinit( "Cdm", sizeof(REAL), (nobs+nsols)*neff2 );
  Csp = (REAL *) zinit( "Csp", sizeof(REAL), (nobs+nsols)*neff2 );

  // expand memory of CPB and CPC to accommodate f2_CPC and f2_CPB
  //  CPB = (REAL *) zinit( "CPB", sizeof(REAL), neff2*neff2 );
  CPB = (REAL *) zinit( "CPB", sizeof(REAL), neff2*neff2*4 );
  //  CPC = (REAL *) zinit( "CPC", sizeof(REAL), ne2h );
  CPC = (REAL *) zinit( "CPC", sizeof(REAL), neff2*(neff2*2+1) );  // includes *2/2

  data = (int *) init( "data", sizeof(int), nobs );
  observed = (REAL *) init( "observed", sizeof(REAL), nei );
  mobserved = (REAL *) init( "mobserved", sizeof(REAL), nEDC );
  galpha = (REAL *) init( "galpha", sizeof(REAL), nei );
  alpha = (REAL *) init( "alpha", sizeof(REAL), nEDC );
  weights = (REAL **) init( "weights", sizeof(REAL *), nindex );
  sum_wts = (REAL *) init( "sum_wts", sizeof(REAL *), nindex );
  scaled_wts = (REAL **) init( "scaled_wts", sizeof(REAL *), nindex );
  trait_wts = (REAL **) init( "trait_wts", sizeof(REAL *), nindex );
  index_h2 = (REAL *) init( "index_h2", sizeof(REAL), nindex );
  nu = (REAL *) init( "nu", sizeof(REAL), noh );
  for (i=0; i<MEAN_VARS; i++) {
    means[i] = (MEAN_STAT *) zinit( "means[i]", sizeof(MEAN_STAT), nEDC );
    means_np[i] = (MEAN_STAT *) zinit( "means_np[i]", sizeof(MEAN_STAT), nEDC );
    means_edc[i] = (MEAN_STAT *) zinit( "means_edc[i]", sizeof(MEAN_STAT), nEDC );
  }
  for (i=0; i<MEAN_VARS_CG; i++) {
    means_cg[i] = (MEAN_STAT *) zinit( "means_cg[i]", sizeof(MEAN_STAT), ntraits );
  }
  trait_label = (char **) zinit( "trait_label", sizeof(char *), ntraits );
  index_label = (char **) zinit( "index_label", sizeof(char *), nEDC );

  for (i=0; i<4; i++) row[i] = element[i] = 0;
  nnn = 1;
  while( fgets( line, BSZ, fpar ) != NULL ) {
    split_line( 0, fnames[PARAMFILE], ++nnn );
    if (!strcmp( value[1], "RR" )) {
      k = 0;
    }else if (!strcmp( value[1], "Gd" )) {
      k = 1;
    }else if (!strcmp( value[1], "Gm" )) {
      k = 1;
    }else if (!strcmp( value[1], "Ed" )) {
      k = 2;
    }else if (!strcmp( value[1], "Em" )) {
      k = 2;
    }else if (!strcmp( value[1], "CG" )) {
      k = 3;
    }else {
      k = 4;
    }

    if (k<4) {
      /* reading matrix info */

      if (nvalues != 2+row[k] && nvalues != 1+norder[k] ) {
	sprintf( msg, "The following line has an unexpected number of arguments (%d vs %d-HS or %d-FS)\n%s",
		 nvalues, 2+row[k], 1+norder[k], line );
	warning( msg );
	error( "You must correct the paramater file to what is expected.\n" );
      }
      for (i=0; i<=row[k]; i++) {
        matrix[k][element[k]] = atof( value[2+i] );
	element[k]++;
      }
      row[k]++;

    }else if (value[1][0] == '#' || !value[1][0]) {
      continue;

    }else if (value[1][0] == 'I') {
      /* reading index info */
      sscanf( value[1], "I%d", &i );
      if (i>0 && i<=nindex) {
	i--;
	if (nvalues < (2 + neffects) )   {
	  sprintf( msg, "The following line has an unexpected number of arguments (%d < %d)\n%s",
		   nvalues, 2+neffects, line );
	  warning( msg );
	  error( "You must correct the paramater file to what is expected.\n" );
	}
	weights[i] = (REAL *) zinit( "weights[i]", sizeof(REAL), neffects );
	scaled_wts[i] = (REAL *) zinit( "scaled_wts[i]", sizeof(REAL), ntraits );
	trait_wts[i] = (REAL *) zinit( "trait_wts[i]", sizeof(REAL), ntraits );
	for (j=0; j<neffects; j++) {
          weights[i][j] = atof( value[2+j] );
	  sum_wts[i] += weights[i][j];
	}
	index_h2[i] = atof( value[2+j] );

	msg[0] = ll = 0;
	for (ofs=3+j; ofs<=nvalues; ofs++) {
	  if (ll) {
	    strcpy( msg+ll, " " );
	    ll += strlen( msg+ll );
	  }
	  strcpy( msg+ll, value[ofs] );
	  ll += strlen( msg+ll );
	}
	if (!ll) {
	  sprintf( msg, "Index_%d", i+1 );
	  ll = strlen( msg );
	}
	if (ll > maxlen_trait) maxlen_trait = ll;
	index_label[i] = (char *) zinit( "index_label[i]", sizeof(char), ll + 1 );
	strcpy( index_label[i], msg );

      }else {
	sprintf( msg, "index I%d is outside the range I1-I%d (%d EDC traits) !!\n", i, nindex, nindex );
	warning( msg );
	sprintf( msg, "You must correct the parameter file (%s).\n", param_file );
	error( msg );
      }

    }else if (!strcmp( value[1], "MODEL" )) {
      /* reading MODEL type */

      if (value[2][0] == 's' || value[2][0] == 'S') {
	if (value[2][1] == 'i' || value[2][1] == 'I') {
	  if (value[2][2] == 'r' || value[2][2] == 'R') {
	    if (value[2][3] == 'e' || value[2][3] == 'E') {
	      sire_model = 1;
	      if (!strncmp( value[2]+4, "AMV", 3 )) animal_model_variances = 1;
	    }
	  }
	}
      }

    }else if (!strcmp( value[1], "OPTION" )) {
      /* reading a run-time option */
      process_option( value[2] );

    }else if (!strcmp( value[1], "TRAITS" )) {
      TRAITS_given = 1;
      if (nvalues != ntraits+1) {
	sprintf( msg, "Number of trait names (%d) does not match number of traits (%d).\n",
		 nvalues-1, ntraits );
	error( msg );
      }
      for (i=0; i<ntraits; i++) {
	trait_label[i] = (char *) init( "trait_label", 
					sizeof(char), sizeof( value[i+2] ) + 1 );
	strcpy( trait_label[i], value[i+2] );
      }

    }else if (value[1][0] == 'F') {
      /* reading a file name */

      if (!strcmp( value[1], "FPED" )) {
	strcpy( fnames[PEDFILE], value[2] );
      }else if (!strcmp( value[1], "FOBS" )) {
	strcpy( fnames[DATAFILE], value[2] );
      }else if (!strcmp( value[1], "FOUT" )) {
	strcpy( fnames[EDCFILE], value[2] );
      }else if (!strcmp( value[1], "FLOG" )) {
	strcpy( fnames[LOGFILE], value[2] );
      }else if (!strcmp( value[1], "FCHK" )) {
	strcpy( fnames[CHKFILE], value[2] );
#ifdef IN_DEVELOPMENT
	runcheck = 1;
#endif
      }else if (!strcmp( value[1], "FSAS" )) {
	strcpy( fnames[SASFILE], value[2] );
      }else if (!strcmp( value[1], "FTRUE" )) {
	strcpy( fnames[TRUFILE], value[2] );
      }else if (!strcmp( value[1], "FEDC_PGM" )) {
	strcpy( edc_pgm, value[2] );
      }else if (!strcmp( value[1], "FEDC_LOG" )) {
	strcpy( edc_log, value[2] );
      }else if (!strcmp( value[1], "FEDC_PAR" )) {
	strcpy( edc_par, value[2] );
      }else if (!strcmp( value[1], "FEDC_OUT" )) {
	strcpy( edc_tmp, value[2] );
      }else if (!strcmp( value[1], "FEDC_SUF" )) {
	strcpy( edc_suf, value[2] );
      }else if (!strcmp( value[1], "FSTDOUT" )) {
	strcpy( fnames[STDOUT], value[2] );
	if (ignore_ETs) strcpy( fnames[STDOUT]+strlen(fnames[STDOUT]), "_noET" );
	redirect_stdout = 1;
      }else if (!strcmp( value[1], "FSTDERR" )) {
	strcpy( fnames[STDERR], value[2] );
	if (ignore_ETs) strcpy( fnames[STDERR]+strlen(fnames[STDERR]), "_noET" );
	redirect_stderr = 1;
      }else {
	sprintf( msg, "%s is not a valid code, correct the parameter file\n", value[1] );
	error( msg );
      }

    }else {
      sprintf( msg, "%s is not a valid code, correct the parameter file\n", value[1] );
      error( msg );
    }

  }

  if (!*fnames[LOGFILE]) redirect_output = redirect_stderr = redirect_stdout = 0;

  for (i=0; i<ntraits; i++) {
    if (!trait_label[i]) {
      sprintf( msg, "Trait%d", i+1 );
      trait_label[i] = (char *) init( "trait_label", 
				      sizeof(char), sizeof( msg ) + 1 );
      strcpy( trait_label[i], msg );
    }
  }

  if (redirect_output && redirect_stderr && strcmp( fname_stderr, fnames[STDERR] )) {
    // in paramater files there was a specification for FSTDERR
    fclose( stderr );
    sprintf( msg, "rm -f %s && mv -f %s %s", fnames[STDERR], fname_stderr, fnames[STDERR] );
    system( msg );
    freopen( fnames[STDERR], "a", stderr );
  }else if (redirect_stderr) {
    freopen( fnames[STDERR], "w", stderr );
  }
  setbuf( stderr, 0 );
  f[STDERR] = stderr;

  if (redirect_output && redirect_stdout && strcmp( fname_stdout, fnames[STDOUT] )) {
    // in paramater files there was a specification for FSTDOUT
    fclose( stdout );
    sprintf( msg, "rm -f %s && mv -f %s %s", fnames[STDOUT], fname_stdout, fnames[STDOUT] );
    system( msg );
    freopen( fnames[STDOUT], "a", stdout );
  }else if (redirect_stdout) {
    freopen( fnames[STDOUT], "w", stdout );
  }
  setbuf( stdout, 0 );
  f[STDOUT] = stdout;

  j = 0;
  for (i=0; i<4; i++) {
    if (element[i] != nelements[i]) {
      if (i > 1 && !element[i]) {
	/* allow for zero E and CG matrices, if not specified */
	for (k=0; k<nelements[i]; k++) matrix[i][k] = 0.0;
      }else {
	sprintf( msg, "incorrect number of elements [%d!=%d] matrix %d [%s]\n",
		 element[i], nelements[i], i, matrix_label[i] );
	warning( msg );
	j = 1;
      }
    }
  }
  for (i=0; i<nindex; i++) {
    if (weights[i] == NULL) {
      sprintf( msg, "weights were not provided for index %d\n", i+1 );
      warning( msg );
    }
  }
  if (j) error( "You must correct the parameter file.\n" );
  /* drop any CR or LF from the end of file names */
  for (i=0; i<NFILES; i++) {
    j = strlen( fnames[i] ) - 1;
    for (k=j; k>-1; i--) {
      if (fnames[i][k] == 13 || fnames[i][k] == 10) {
	fnames[i][k] = 0;
      }else {
	break;
      }
    }
  }
  return;
}

//**********************************************************************
void read_ped() {
  // process pedigree file
  int i, j, n, np, garbage, *ped_stored;
  REAL c;
  
  /* find max ID in pedigree file */
  nanml = nnn = 0;
  while( fgets( line, BSZ, fped ) != NULL) {
    split_lineB( nflds[PEDFILE], fnames[PEDFILE], ++nnn );
    for (i=1; i<=3; i++) {
      n = atoi( value[i] );
      if (n > nanml) nanml = n;
    }
  }
  rewind( fped );
  
  /* check data file as well, in case foster IDs > max pedigree ID */
  nnn = 0;
  while( fgets( line, BSZ, frec ) != NULL ) {
    split_line( nflds[DATAFILE], fnames[DATAFILE], ++nnn );
    an = atoi( value[1] );
    if (an < 1) {
      warning( line );
      sprintf( msg, "Invalid animal [%d] with performance record, animal numbers must be greater than zero\n", an );
      warning( msg );
//      error( msg );
    }
    if (an > nanml) nanml = an;
    if (nmaternal) {
      dam = atoi( value[2] );
      if (dam > nanml) nanml = dam;
    }
  }
  rewind( frec );
  
  ped = (PED *) zinit( "ped", sizeof(PED), (nanml+1) );
  ped_stored = (int *) zinit( "ped_stored", sizeof(int), (nanml+1) );

  //  for (pp=ped+1; pp<=ped+nanml; pp++) pp->adiag = 1.0;
  n = np = nnn = 0;
  fprintf( flog, "\nFirst %d pedigrees with known parent(s)...\n", NPRINT_PED );
  while( fgets( line, BSZ, fped ) != NULL) {
    split_lineB( nflds[PEDFILE], fnames[PEDFILE], ++nnn );
    if ((i = atoi( value[1] ))) {
      if (!i) {
	warning( "  PEDIGREE: " );
	warning( line );
	error( "ERROR: Animal number 0 IS NOT ALLOWED.  IDs must be 1-based\n" );
      }
      if (ped_stored[i]) {
	warning( "  PEDIGREE: " );
	warning( line );
	error( "ERROR: More than 1 pedigree record for above animal (NOT ALLOWED).\n" );
      }
      ped_stored[i] = 1;
      pp = ped + i;
      pp->sire = atoi( value[2] );
      if (pp->sire && ped[ pp->sire ].sex == FEMALE) {
	sprintf( msg, "Sire %d of Progeny %d is also a Dam !!\n", pp->sire, (int) (pp-ped) );
	warning( msg );
      }
      ped[ pp->sire ].sex = MALE;
      pp->dam = atoi( value[3] );
      if (pp->dam && ped[ pp->dam ].sex == MALE && !sire_model) {
	sprintf( msg, "Dam %d of Progeny %d is also a Sire !!\n", pp->dam, (int) (pp-ped) );
	warning( msg );
      }
      if (sire_model) {
	ped[ pp->dam ].sex = MALE;
      }else {
	ped[ pp->dam ].sex = FEMALE;
      }
      if (pp->sire > nanml || pp->dam > nanml) {
	sprintf( msg, "found parent with ID > number of pedigrees (A=%d S=%d D=%d)\n", 
		 i, pp->sire, pp->dam );
        warning( msg );
        warning( " ... expecting pedigree record for all animals" );
        error( " including parents, sequentially in file\n" );
      }
      if (pp->sire || pp->dam) {
	if (np<NPRINT_PED) {
	  fputs( line, flog );
	  np++;
	}
      }
      /*
      if (sire_model) {
	c = 16.0 / 11 + 4*(!pp->sire) + (!pp->dam);
	if (pp->dam) ped[ pp->dam ].adiag += c * 0.0625;
      }else {
	c = 4.0 / 2 + (!pp->sire) + (!pp->dam);
	if (pp->dam) ped[ pp->dam ].adiag += c * 0.25;
      }
      pp->adiag += c - 1.0;
      if (pp->sire) ped[ pp->sire ].adiag += c * 0.25;
      */
    }
  }
  j = 0;
  for (i=1; i<=nanml; i++) {
    if (!ped_stored[i]) {
      if (j<10) {
	fprintf( flog, "No pedigree record for animal %d, assuming unknown parents\n", i );
      }
      j++;
    }
  }
  if (j) {
    fprintf( flog, "WARNING: %d animals did not have a pedigree record\n", j );
  }
  fprintf( flog, "\n%d animals in the pedigree\n", nanml );
  free( ped_stored );
  return;
}

//**********************************************************************
void split_line( int flds_expected, char *fname, int rec_num ) {
  // parse function to extract fields from a delimited string
  char *c;
  int len=0, i;
  static char *delims=" \t,";
  while( line[len] && line[len] != 13 && line[len] != '\n') {
    split_up_line[len] = line[len];
    len++;
  }
  split_up_line[len] = 0;
  nvalues = 0;
  c = split_up_line;
  while(c) {
    value[ nvalues+1 ] = strsep( &c, delims );
    if ( (*value[ nvalues+1 ]) ) nvalues++;
  }
  if (flds_expected && flds_expected != nvalues) {
    sprintf( msg, "Expected %d values, but found %d", flds_expected, nvalues );
    warning( msg );
    if (fname[0]) {
      sprintf( msg, ", File=[%s]", fname );
      warning( msg );
    }
    if (rec_num) {
      sprintf( msg, ", record #%d", rec_num );
      warning( msg );
    }
    if (nvalues) {
      warning( " => " );
      for (i=0; i<nvalues; i++) {
	if (i) {
	  warning( "," );
	}else {
	  warning( "[" );
	}
	warning( value[i+1] );
      }
      warning( "]" );
    }
    warning( ".\n" );
    sprintf( msg, "\n%s\n", line );
    error( msg );
  }
  return;
}

//**********************************************************************
void split_lineB(  int flds_expected, char *fname, int rec_num ) {
  // parse function to extract fields from a delimited string
  char *c;
  int len=0, i;
  static char *delims=" \t,";
  while( line[len] && line[len] != 13 && line[len] != '\n') {
    split_up_line[len] = line[len];
    len++;
  }
  split_up_line[len] = 0;
  nvalues = 0;
  c = split_up_line;
  while(c) {
    value[ nvalues+1 ] = strsep( &c, delims );
    if ( (*value[ nvalues+1 ]) ) nvalues++;
  }
  if (flds_expected && flds_expected > nvalues) {
    sprintf( msg, "Expected %d or more values, but found %d", flds_expected, nvalues );
    warning( msg );
    if (fname[0]) {
      sprintf( msg, ", File=[%s]", fname );
      warning( msg );
    }
    if (rec_num) {
      sprintf( msg, ", record #%d", rec_num );
      warning( msg );
    }
    if (nvalues) {
      warning( " => " );
      for (i=0; i<nvalues; i++) {
	if (i) {
	  warning( "," );
	}else {
	  warning( "[" );
	}
	warning( value[i+1] );
      }
      warning( "]" );
    }
    warning( ".\n" );
    sprintf( msg, "\n%s\n", line );
    error( msg );
  }
  return;
}

//**********************************************************************
void summarize_means( MEAN_STAT *mm[MEAN_VARS], int mean_vars, char *condition ) {
  // write to log file: MEAN statistics for EDCs and comparative counts
  int i, j, k, ll;
  MEAN_STAT *m;
  REAL x;
  char tmp[BSZ];
  fprintf( flog, "\n" );
  fprintf( flog, "Summary of descriptive means (%s)\n", condition );
  fprintf( flog, "----------------------------" );
  j = strlen( condition ) + 3;
  for (i=0; i<j; i++ ) fprintf( flog, "-" );
  fprintf( flog, "\n\n" );
  fprintf( flog, "EDC_trait" );
  for (k=strlen("EDC_trait"); k<LEN_TRAIT; k++) fprintf( flog, " " );
  fprintf( flog, "Variable" );
  for (k=strlen("Variable"); k<LEN_LABEL; k++) fprintf( flog, " " );
  fprintf( flog, "         n      Mean     Std_Dev       Min        Max \n" );
  for (i=0; i<nindex; i++) {
    fprintf( flog, "\n" );
    ll = strlen( index_label[i] );
    if (ll >= LEN_TRAIT) {
      fprintf( flog, "%s\n", index_label[i] );
      strncpy( tmp, index_label[i], LEN_TRAIT-5 );
      sprintf( tmp+LEN_TRAIT-5, "..%s", index_label[i]+ll-2 );
      ll = strlen( tmp );
    }else {
      strcpy( tmp, index_label[i] );
    }
    for (j=0; j<mean_vars; j++) {
      fprintf( flog, "%s", tmp );
      for (k=ll; k<LEN_TRAIT; k++) fprintf( flog, " " );
      fprintf( flog, "%s", mean_label[j] );
      for (k=strlen(mean_label[j]); k<LEN_LABEL; k++) fprintf( flog, " " );
      m = mm[j]+i;
      if (m->n > 1) {
        x = sqrt( m->ss / ( m->n - 1 ) );
      }else {
        x = 0.0;
      }
      fprintf( flog, "%10d%11.4f%11.4f%11.3f%11.3f\n",
                     m->n, m->u, x, m->min, m->max );
    }
  }
  return;
}

//**********************************************************************
void summarize_means_cg() {
  // write to log file: MEAN statistics for size and structure of CGs
  int i, j, k, nv, ll;
  MEAN_STAT *m;
  REAL x;
  char tmp[BSZ];
  fprintf( flog, "\n" );
  fprintf( flog, "Structure of contemporary groups\n" );
  fprintf( flog, "--------------------------------\n\n" );
  fprintf( flog, "Trait" );
  for (k=strlen("Trait"); k<LEN_TRAIT; k++) fprintf( flog, " " );
  fprintf( flog, "Counts" );
  for (k=strlen("Counts"); k<LEN_LABEL; k++) fprintf( flog, " " );
  fprintf( flog, "       nCG      Mean     Std_Dev       Min        Max \n" );
  for (i=0; i<ntraits; i++) {
    fprintf( flog, "\n" );
    if (i<nmaternal) {
      nv = MEAN_VARS_CG;
    }else {
      nv = MEAN_VARS_CG - 1;
    }
    ll = strlen( trait_label[i] );
    if (ll >= LEN_TRAIT) {
      fprintf( flog, "%s\n", trait_label[i] );
      strncpy( tmp, trait_label[i], LEN_TRAIT-5 );
      sprintf( tmp+LEN_TRAIT-5, "..%s", trait_label[i]+ll-2 );
      ll = strlen( tmp );
    }else {
      strcpy( tmp, trait_label[i] );
    }
    for (j=0; j<nv; j++) {
      fprintf( flog, "%s", tmp );
      for (k=ll; k<LEN_TRAIT; k++) fprintf( flog, " " );
      fprintf( flog, "%s", mean_label_cg[j] );
      for (k=strlen(mean_label_cg[j]); k<LEN_LABEL; k++) fprintf( flog, " " );
      m = means_cg[j]+i;
      if (m->n > 1) {
        x = sqrt( m->ss / ( m->n - 1 ) );
      }else {
        x = 0.0;
      }
      fprintf( flog, "%10d%11.4f%11.4f%11.3f%11.3f\n",
                     m->n, m->u, x, m->min, m->max );
    }
  }
  return;
}

//**********************************************************************
void summarize_parameters() {
  // write to log file: input parameters and derived parameters
  // check input matrices are p.s.d., heritabilities, etc.
  // derive covariances and parameters required for the approximations
  int i, j, k, n, ii, im, jm, ij, jk, dd, md, mm, ll, itmp[BSZ], *PDlist;
  int flag, errors, PDerrors, *h2_error;
  REAL x, z, *index_h2_calc;

  if (n_include_traits) {
    // this is a test run on a subset of traits, zero out covariances of excluded traits
#ifdef DEBUG
    printf( "Matrix G, before deleting the ignored traits\n" );
    for (i=k=0; i<neffects; i++) {
      for (j=0; j<=i; j++,k++) printf( "%8.3f", G[k] );
      printf( "\n" );
    }
#endif
    for (i=0; i<ntraits; i++) {
      for (j=0; j<ntraits; j++) if (!including_trait(i) || !including_trait(j)) {
	R[irc(i,j)] = 0.0;
	if (CGvar) CG[irc(i,j)] = 0.0;
	G[irc(i,j)] = 0.0;
	E[irc(i,j)] = 0.0;
	if (i<nmaternal) {
	  G[irc(i+ntraits,j)] = 0.0;
	  E[irc(i+ntraits,j)] = 0.0;
	  if (j<nmaternal) {
	    G[irc(i+ntraits,j+ntraits)] = 0.0;
	    E[irc(i+ntraits,j+ntraits)] = 0.0;
	  }
	}
      }
    }
#ifdef DEBUG
    printf( "Matrix G, after deleting the ignored traits\n" );
    for (i=k=0; i<neffects; i++) {
      for (j=0; j<=i; j++,k++) printf( "%8.3f", G[k] );
      printf( "\n" );
    }
#endif
  }

  errors = 0;
  if (!flog) {
    errors = 1;
    flog = stdout;
  }
  fputs( cmdline, flog );
  fprintf( flog, "Output from crEDC version %s\n", version );
  fprintf( flog, "================================\n\n" );
  fprintf( flog, "%s\n", EDC_method );
  fprintf( flog, "*****************************************\n" );
  if (sire_model) {
    fprintf( flog, "*  NOTE: A SIRE MODEL was assumed       *\n" );
  }else {
    fprintf( flog, "*  NOTE: An ANIMAL MODEL was assumed    *\n" );
  }
  if (!sire_model && nmaternal) {
    if (maternal_only) {
      fprintf( flog, "*   MACE MGS => maternal only           *\n" );
    }else {
      fprintf( flog, "*   MACE MGS => maternal plus direct/2  *\n" );
    }
  }
  if (ignore_h2_check) fprintf( flog, "*   Ignoring heritability check         *\n" );
  if (ignore_PD_check) fprintf( flog, "*   Ignoring positive definite checks   *\n" );
  if (ignore_MAXSIRES_check) fprintf( flog, "*    Ignoring MAXSIRE check             *\n" );
  if (ignore_WTS_check) fprintf( flog, "*   Ignoring record weights checks      *\n" );
  if (h2_within_CG)    fprintf( flog, "*   Ignoring V(CG) for h2 calculation   *\n" ); 
  fprintf( flog, "*****************************************\n\n" );

  fprintf( flog, "Expected Input File Contents:\n" );
  fprintf( flog, "=============================\n" );
  fprintf( flog, "\n  FPED [%s]\n", fnames[PEDFILE] );
  if (sire_model) {
    fprintf( flog, "    bull sire mgs\n" );
  }else {
    fprintf( flog, "    animal sire dam\n" );
  }
  fprintf( flog, "\n  FOBS [%s]\n", fnames[DATAFILE] );
  for (k=0; k<2; k++) {
    if (k) {
      fprintf( flog, "\nOR " );
    }else {
      fprintf( flog, "   " );
    }
    if (sire_model) {
      fprintf( flog, "sire" );
      if (nmaternal) fprintf( flog, " mgs" );
    }else {
      fprintf( flog, "animal" );
      if (nmaternal) fprintf( flog, " dam" );
    }
    log_list( ntraits, "CG" );
  }
  log_list( ntraits, "wt" );
  fprintf( flog, "\n" );
  fprintf( flog, "where CG(i) is the contemporary group and wt(i) the record weight for trait i\n" );  
  fprintf( flog, "\n" );
  fprintf( flog, "Output File Contents:\n" );
  fprintf( flog, "=====================\n" );
  fprintf( flog, "\n  FOUT [%s]\n", fnames[EDCFILE] );
  if (RELIABILITIES) {
    fprintf( flog, "   animal" );
    log_list( neffects+nEDC, "Rel" );
    fprintf( flog, "\n" );
    for (i=0; i<ntraits; i++) {
      fprintf( flog, "    Rel%d = reliability for %s\n", i+1, trait_label[i] );
    }
    for (i=0; i<nmaternal; i++) {
      fprintf( flog, "    Rel%d = reliability for %s maternal\n", ntraits+i+1, trait_label[i] );
    }
    for (i=0; i<nEDC; i++) {
      fprintf( flog, "    Rel%d = reliability for %s\n", neffects+i+1, index_label[i] );
    }
    if (include_header) {
      fprintf( fout, "animal" );
      for (i=0; i<ntraits; i++) fprintf( fout, ",%s_%s", rel_lbl, trait_label[i] );
      for (i=0; i<nmaternal; i++) fprintf( fout, ",%s_Mat_%s", rel_lbl, trait_label[i] );
      for (i=0; i<nEDC; i++) fprintf( fout, ",%s_%s", rel_lbl, index_label[i] );
      fprintf( fout, "\n" );
      fflush( fout );
    }
  }else {
    fprintf( flog, "   sire" );
    log_list( nEDC, "EDC" );
    log_list( nEDC, "ND" );
    if (count_grandprogeny) {
      log_list( nEDC, "Npgp" );
      log_list( nEDC, "Nmgp" );
    }
    if (edc_reliabilities) {
      log_list( nEDC, "Rel" );
    }
    fprintf( flog, "\n" );
    fprintf( flog, "where ND(i) is the number of progeny records for EDC trait i\n" );
    if (count_grandprogeny) {
      fprintf( flog, "where Npgp(i) is the number of grandprogeny records through sons for EDC trait i\n" );
      fprintf( flog, "where Nmgp(i) is the number of grandprogeny records through daughters for EDC trait i\n" );
    }
    if (edc_reliabilities) {
      fprintf( flog, "where Rel(i) is the reliability approximated from EDC(i)\n" );
    }
    if (include_header) {
      fprintf( fout, "sire" );
      for (i=0; i<nEDC; i++) fprintf( fout, ",EDC_%s", index_label[i] );
      for (i=0; i<nEDC; i++) fprintf( fout, ",ND_%s", index_label[i] );
      if (edc_reliabilities) {
	for (i=0; i<nEDC; i++) fprintf( fout, ",Rel_%s", index_label[i] );
      }
      fprintf( fout, "\n" );
      fflush( fout );
    }
  }
  fprintf( flog, "\n" );
  fprintf( flog, "--------------------------------------------------------------------\n" );
  fprintf( flog, "\n" );
  if (errors) flog = 0;

  if (file_access_error) {
    if (flog) {
      sprintf( msg, "\nSee FLOG [%s] for helpful information.\n", fnames[LOGFILE] );
      warning( msg );
    }
    warning( "\n" );
    if (!fped) warning( "!! FPED must be specified in the parameter file.\n" );
    if (!frec) warning( "!! FOBS must be specified in the parameter file.\n" );
    if (!flog) warning( "!! FLOG must be specified in the parameter file.\n" );
    error( "***** Stopping due to file access error(s) ******\n" );
  }

  h2_error = (int *) zinit( "h2_error", sizeof(int), nindex );
  index_h2_calc = (REAL *) init( "index_h2_calc", sizeof(REAL), nindex );

  k = 0;
  for (i=0; i<ntraits; i++) {
    for (j=0; j<=i; j++) {
      vary[k] = G[k] + R[k] + E[k];
      varg[k] = G[k];
      k++;
    }
  }

  CGvar = 0;
  for (i=0; !CGvar && i<ntraits; i++) {
    if (CG[irc(i,i)]) CGvar = 1;
  }
  if (CGvar && !h2_within_CG) {
    //    for (k=0; k<nth; k++) vary[k] += CG[k];
    for (i=0; i<ntraits; i++) vary[irc(i,i)] += CG[irc(i,i)];
  }

  for (i=0; i<nmaternal; i++) {
    im = i + ntraits;
    for (j=0; j<=i; j++) {
      jm = j + ntraits;
      ij = irc(im,jm);
      k = irc(i,j);
      // account for an-dam relationship = 0.5
      x = 0.5 * (G[irc(i,jm)] + G[irc(j,im)]);
      vary[k] += x + G[ij] + E[ij];
      varg[k] += x + G[ij];
    }
    for (; j<ntraits; j++) {
      k = irc(i,j);
      x = 0.5 * G[irc(j,im)];
      vary[k] += x;
      varg[k] += x;
    }
  }

  if (sire_model && animal_model_variances) convert_AMV_to_SMV();

  for (i=0; i<nth; i++) R2[i] = R[i];
  for (ii=0; ii<nmaternal; ii++) {
    for (j=0; j<ntraits; j++) {
      R2[i] = 0.0;
      i++;
    }
    for (; j<neffects; j++) {
      R2[i] = R[irc(ii,j-ntraits)];
      i++;
    }
  }

  PDerrors = errors = flag = ncomb = 0;
  for (i=0; i<nt2p; i++) {
    combination_inc[i] = 0;
    for (j=jk=0; j<ntraits; j++) {
      for (k=0; k<=j; k++,jk++) {
	if ((i&(1<<j)) && (i&(1<<k))) {
	  Rinv_tmp[jk] = R[jk];
	}else {
	  Rinv_tmp[jk] = 0.0;
	}
      }
    }

    if (ignore_resid_correlations) {
      for (j=0; j<ntraits; j++) {
	for (k=0; k<j; k++) Rinv_tmp[irc(j,k)] = 0.0;
      }
    }

    if (invert( Rinv_tmp, ntraits ) < 0) {
      PDerrors = 1;
      for (j=0; j<ntraits; j++) {
	if (Rinv_tmp[irc(j,j)] < 0) {
	  // flag suspect traits in this trait combination
	  combination_inc[i] |= (1<<j);
	}
      }
      // flag all traits suspected in one or more trait combinations
      if (combination_inc[i]) {
	errors |= combination_inc[i];
      }else {
	// combo is neg. definite but not sure which traits to suspect
	combination_inc[i] = -1;
      }
    }

    for (j=Rinv_n[i]=0; j<nth; j++) if (Rinv_tmp[j]) Rinv_n[i]++;
    if (Rinv_n[i]) {
      Rinv[i] = (REAL *) init( "Rinv[i]", sizeof(REAL), Rinv_n[i] );
      Rinv_ij[i] = (int *) init( "Rinv_ij[i]", sizeof(int), Rinv_n[i] );
      for (j=n=0; j<nth; j++) {
	if (Rinv_tmp[j]) {
	  Rinv_ij[i][n] = j;
	  Rinv[i][n] = Rinv_tmp[j];
	  n++;
	}
      }
    }
  }

  if (PDerrors) {
    // i & (~errors) => the combination includes irrelevant traits
    warning( "!! Matrix R is not positive definite, which could cause negative EDCs.\n" );
    for (i=ncomb=0; i<nt2p; i++) {
      if (combination_inc[i]<0 || (combination_inc[i]>0 && !(i&(~errors)))) ncomb++;
    }
    if (ncomb) {
      PDlist = (int *) init( "PDlist", sizeof(int), ncomb );
      for (i=j=0; i<nt2p; i++) {
	if (combination_inc[i]<0 || (combination_inc[i]>0 && !(i&(~errors)))) {
	  PDlist[j] = i;
	  j++;
	}
      }
      if (j != ncomb) {
	sprintf( msg, "Very unexpected error for PDlist: [j=%d] != [ncomb=%d].\n", j, ncomb );
	error( msg );
      }
      warning( "!!   Negative-definite sub-matrices in R are:\n" );
      for (j=ll=0; j<ntraits; j++) if (ll<(k=strlen( trait_label[j] ))) ll = k;
      for (j=flag=0; j<ntraits; j++) {
	if ((1<<j)&errors) {
	  flag = 1;
	  sprintf( msg, "%s", cfmt( trait_label[j], ll ) );
	  warning( msg );
	  for (k=0; k<ncomb; k++) {
	    if (check_bit( j, PDlist[k] )) {
	      if (check_bit( j, combination_inc[ PDlist[k] ] )) {
		sprintf( msg, "  X" );
	      }else {
		sprintf( msg, "  X" );
	      }
	    }else {
	      sprintf( msg, "   " );
	    }
	    warning( msg );
	  }
	  warning( "\n" );
	}
      }
    }
    if (!flag) {
      // a suspect trait list could not be identified
      warning( "  .... ? Sorry, I was not able to narrow it down for you.\n" );
    }
    warning( "\n" );
    for (i=0; i<nth; i++) Rinv_tmp[i] = R[i];
    invert( Rinv_tmp, ntraits );
    pr( flog, "Inverse of matrix R", Rinv_tmp, nth );
    warning( "\n" );
  }

  // check if G matrix is positive-semi-definite
  for (i=0; i<neh; i++) G2[i] = G[i];
  for (; i<nsh; i++) G2[i] = 0.0;
#ifdef DEBUG
  sprintf( msg, "G-matrix" );
  pr( stdout, msg, G2, neh );
#endif
  if ((i=invert( G2, neffects )) < 0) {
    PDerrors = 1;
    warning( "!! Matrix G is not positive definite, which could cause negative EDCs.\n" );
    warning( "!!  Suspect traits/effects include:" );
    for (j=flag=0; j<ntraits; j++) {
      if (G2[irc(j,j)] < 0) {
	flag = 1;
	if (nmaternal) {
	  sprintf( msg, ", %s direct", trait_label[j] );
	}else {
	  sprintf( msg, ", %s", trait_label[j] );
	}
	warning( msg );
      }
    }
    for (j=0; j<nmaternal; j++) {
      if (G2[irc(ntraits+j,ntraits+j)] < 0) {
	flag = 1;
	sprintf( msg, ", %s maternal", trait_label[j] );
	warning( msg );
      }
    }
    if (!flag) {
      // a suspect trait list could not be identified
      warning( "  .... ? Sorry, I was not able to narrow it down for you.\n" );
    }
    warning( "\n" );
    pr( flog, "Inverse of matrix G", G2, neh );
    warning( "\n" );
  }
#ifdef DEBUG
  sprintf( msg, "G-inverse" );
  pr( stdout, msg, G2, neh );
#endif

  if (sire_model) {
    for (i=0; i<neh; i++) S[i] = G[i];
  }else {
    ij = 0;
    for (i=0; i<ntraits; i++) {
      for (j=0; j<=i; j++) {
        S[ij] = 0.25 * G[ij];
        ij++;
      }
    }
    for (i=0; i<nmaternal; i++) {
      im = i + ntraits;
      for (j=0; j<=i; j++) {
        jm = j + ntraits;
        dd = irc(i,j);
        md = irc(im,j);
        mm = irc(im,jm);
	if (maternal_only) {
	  // MGS maternal contribution
	  S[mm] = 0.25 * G[mm];
	}else {
	  // MGS total genetic contribution
	  S[mm] = 0.25 * (G[mm] + G[md] + 0.25 * G[dd]);
	}
      }
      for (j=0; j<ntraits; j++) {
        jm = j + ntraits;
        md = irc(im,j);
	if (maternal_only) {
	  // MGS maternal contribution
	  S[md] = 0.25 * G[md];
	}else {
	  // MGS total genetic contribution
	  S[md] = 0.25 * (G[md] + 0.5 * G[dd]);
	}
      }
    }
  }
  // NOTE: assuming sire and mgs are unrelated (a(sire,mgs)=0), else normally close to 0
  //       therefore no covariance contribution to vary (a(sire,mgs) * 2 * S(sire,mgs) = 0)
  //       and vary = S(sire,sire) + S(mgs,mgs) + RS 
  for (i=0; i<nth; i++) RS[i] = vary[i] - S[i];
  for (i=0; i<nmaternal; i++) {
    for (j=0; j<=i; j++) {
      RS[irc(i,j)] -= S[irc(i+ntraits,j+ntraits)];
    }
  }
  
  /* NOTE: 'RS' is for the full model: y=D+M+e, 
         BUT 'alpha' is for partial models for ST-MACE: y=D+e, y=M+e */
  for (i=0; i<ntraits; i++) {
    ii = irc(i,i);
    if (S[ii]) {
      galpha[i] = (vary[ii] - S[ii]) / (S[ii]);
    }else {
      galpha[j] = 0.0;
    }
    if (i<nmaternal) {
      j = i+ntraits;
      k = irc(j,j);
      if (S[k]) {
	galpha[j] = (vary[ii] - S[k]) / (S[k]);
      }else {
	galpha[j] = 0.0;
      }
    }
  }
  // Indexes could include D and M, in which case re-scaling may be needed.
  // Must determine proportion of total genetic variance, by trait, and re-scale
  //   within versus across traits to get correct heritability of the index
  for (k=0; k<nindex; k++) {
    for (i=0; i<nmaternal; i++) {
      im = i+ntraits;
      if (weights[k][im]) {
	// working with dam contribution to the observation
	z = weights[k][i] / 0.5;  // direct contribution from the dam = 0.5
	if (weights[k][im] > z) {
	  z = weights[k][im];   // maternal contribution from the dam = 1
	}
	// use weights(D,M)/z to determine proportion of total maternal contribution included
	// and use z as the trait weight in the index
	C[0] = weights[k][i] / z;
	C[1] = weights[k][im] / z;
	P[0] = G[ irc(i,i) ];
	P[1] = G[ irc(i,im) ];
	P[2] = G[ irc(im,im) ];
	x = kbk( C, P, 2 );
      }else {
	// working with animal contribution to the observation
	z = weights[k][i];
	x = G[ irc(i,i) ];
      }
      // trait weight * sqrt( proportion of total genetic variance included for trait)
      scaled_wts[k][i] = z * sqrt(x / varg[irc(i,i)]);
      trait_wts[k][i] = z;
    }
    for (; i<ntraits; i++) {
      // total genetic variance is for direct effect only
      trait_wts[k][i] = scaled_wts[k][i] = weights[k][i];  // = z * 1.0;
    }
  }

  sprintf( fmt, "%%%ds", maxlen_trait );
  fprintf( flog, fmt, "Effect" );
  fprintf( flog, "  Heritability  Variance ratio\n" );
  for( i=0; i<ntraits; i++) {
    if (galpha[i]) {
      x = 4.0 / ( 1.0 + galpha[i] );
    }else {
      x = 0.0;
    }
    sprintf( msg, " Dir_g%d", i+1 );
    fprintf( flog, fmt, msg );
    fprintf( flog, "%11.4f%11.3f (Sire)", x, galpha[i] );
    if (TRAITS_given) fprintf( flog, "   %s", trait_label[i] );
    fprintf( flog, "\n" );
  }
  for( i=0; i<nmaternal; i++) {
    im = i + ntraits;
    if (galpha[im]) {
      x = 4.0 / ( 1.0 + galpha[im] );
    }else {
      x = 0.0;
    }
    z = G[irc(im,im)] / vary[irc(i,i)];
    if (sire_model) z *= 4;
    sprintf( msg, " Mat_g%d", i+1 );
    fprintf( flog, fmt, msg );
    fprintf( flog, "%11.4f", z );
    if (sire_model || maternal_only) {
      fprintf( flog, "%11.3f (MGS) ", galpha[im] );
    }else {
      fprintf( flog, "\n" );
      sprintf( msg, "  M+G/2" );
      fprintf( flog, fmt, msg );
      fprintf( flog, "%11.4f%11.3f (MGS) ", x, galpha[im] );
    }
    if (TRAITS_given) fprintf( flog, "   %s maternal", trait_label[i] );
    fprintf( flog, "\n" );
  }
  fprintf( flog, "\n" );
  fprintf( flog, "%6s", " " );
  fprintf( flog, fmt, "EDC_trait" );
  fprintf( flog, "      MACE      Variance   K'      " );
  for (i=1; i<ntraits; i++) fprintf( flog, "%8s", " " );
  if (nmaternal) fprintf( flog, "maternal" );
  fprintf( flog, "\n" );
  fprintf( flog, "%6s", " " );
  fprintf( flog, fmt, "(=K'g)" );
  fprintf( flog, "  Heritability    ratio    " );
  for (i=0; i<ntraits; i++) {
    ll = strlen( trait_label[i] );
    for (j=0; j<7 && j<ll; j++) fprintf( flog, "%c", trait_label[i][j] );
    for (; j<8; j++) fprintf( flog, " " );
  }
  for (i=0; i<nmaternal; i++) {
    ll = strlen( trait_label[i] );
    for (j=0; j<7 && j<ll; j++) fprintf( flog, "%c", trait_label[i][j] );
    for (; j<8; j++) fprintf( flog, " " );
  }
  fprintf( flog, "\n" );
  errors = 0;
  for( i=0; i<nindex; i++) {
    if (index_h2[i]) {
      alpha[i] = (4.0 - index_h2[i]) / index_h2[i];
    }else {
      alpha[i] = 0.0;
    }
    x = kbk( scaled_wts[i], varg, ntraits );
    z = x / kbk( trait_wts[i], vary, ntraits );
    if (sire_model) z *= 4;
    //fprintf( flog, "h2_calc = %f / %f = %f\n", x, kbk( trait_wts[i], vary, ntraits ), z );
    x = z - index_h2[i];
    if (x<0) x=-x;
    if (x > .01) {
      errors = 1;
      h2_error[i] = 1;
      index_h2_calc[i] = z;
      fprintf( flog, "?" );
    }else {
      fprintf( flog, " " );
    }
    if (i<9) {
      fprintf( flog, " I%d) ", i+1 );
    }else {
      fprintf( flog, "I%d) ", i+1 );
    }
    fprintf( flog, fmt, index_label[i] );
    fprintf( flog, "%11.4f", index_h2[i] );
    if (h2_error[i]) {
      fprintf( flog, "*" );
    }else {
      fprintf( flog, " " );
    }
    fprintf( flog, "%11.3f  ", alpha[i] ); 
    for (j=0; j<neffects; j++) fprintf( flog, "%8.4f", weights[i][j] );
    fprintf( flog, "\n" );
  }

  if (errors) {
    fprintf( flog, "\n" );
    fprintf( flog, "*NOTE: Heritability does not match covariance parameters provided.\n" );
    fprintf( flog, "\n" );
    for (i=0; i<66; i++) fprintf( flog, "=" );
    fprintf( flog, "\n" );
    fprintf( flog, "Suggested heritabilities calculated from covariances provided\n" );
    for (i=0; i<66; i++) fprintf( flog, "-" );
    fprintf( flog, "\n" );
    fprintf( flog, "                          Heritability\n" );
    fprintf( flog, "         EDC_trait   Provided   Calculated from parameters\n" );
    for (i=0; i<66; i++) fprintf( flog, "-" );
    fprintf( flog, "\n" );
    for (i=0; i<nindex; i++) {
      if (h2_error[i]) {
	fprintf( flog, " " );
	if (i<9) {
	  fprintf( flog, " I%d) ", i+1 );
	}else {
	  fprintf( flog, "I%d) ", i+1 );
	}
	fprintf( flog, fmt, index_label[i] );
	fprintf( flog, "%11.4f*", index_h2[i] );
	fprintf( flog, "%11.4f", index_h2_calc[i] );
	fprintf( flog, "\n" );
	if (!ignore_h2_check) {
	  fprintf( stderr, "ERROR: Heritability for I%d (%s) does not match parameters.\n",
		   i+1, index_label[i] );
	  fprintf( stderr, "       Change it from %.4f to %.4f",
		   index_h2[i], index_h2_calc[i] );
	  fprintf( stderr, ", or add 'OPTION ignore_h2_check' to parameter file.\n" );
	}
      }
    }
    for (i=0; i<66; i++) fprintf( flog, "=" );
    fprintf( flog, "\n" );
    if (!ignore_h2_check) {
      fprintf( flog, "\n" );
      fprintf( flog, "Program STOPPED due to heritability check.\n" );
      fprintf( flog, "  Correct heritabilities OR add 'OPTION ignore_h2_check' in parameter file\n" );
      error( "" );
    }
  }

  fprintf( flog, "\n" );                  
  fprintf( flog, "R (correlations above diagonal) ...\n" );
  fprintf( flog, " " );
  for (i=0; i<ntraits; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
  fprintf( flog, "\n" );

  for (i=0; i<ntraits; i++) {
    x = R[irc(i,i)];
    if (x<0) x = -x;
    // count number of integer digits (store as itmp[i])
    j = 0;
    while( x >= 1 ) {
      j++;
      x *= .1;
    }
    itmp[i] = j; 
  }
  k = 0;
  for (i=0; i<ntraits; i++) {
    for (j=0; j<=i; j++) {
      // determine number of integer digits required (ll) and set format
      ll = (itmp[i] + itmp[j]) / 2;
      if (ll>8) {
	ll = 8;
      }else if (ll<3) {
	ll = 3;
      }
      sprintf( fmt, "%%11.%df", 8 - ll );
      fprintf( flog, fmt, R[k] );
      k++;
    }
    fprintf( flog, "\\" );
    for (; j<ntraits; j++) {
      if (R[irc(i,i)] && R[irc(j,j)]) {
	fprintf( flog, "%11.3f", R[irc(i,j)] / sqrt(R[irc(i,i)]*R[irc(j,j)]));
      }else {
	fprintf( flog, "       .   " );
      }
    }
    fprintf( flog, "\n" );
  }

  fprintf( flog, "\n" );
  fprintf( flog, "G (correlations above diagonal) ...\n" );
  if (nmaternal) {
    for (i=0; i<ntraits; i++) fprintf( flog, "%11s", " " );
    fprintf( flog, "%s", cfmt( "maternal", 11 ) );
    fprintf( flog, "\n" );
  }
  fprintf( flog, " " );
  for (i=0; i<ntraits; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
  for (i=0; i<nmaternal; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
  fprintf( flog, "\n" );

  for (i=0; i<neffects; i++) {
    x = G[irc(i,i)];
    if (x<0) x = -x;
    // count number of integer digits (store as itmp[i])
    j = 0;
    while( x >= 1 ) {
      j++;
      x *= .1;
    }
    itmp[i] = j; 
  }
  k = 0;
  for (i=0; i<neffects; i++) {
    for (j=0; j<=i; j++) {
      // determine number of integer digits required (ll) and set format
      ll = (itmp[i] + itmp[j]) / 2;
      if (ll>8) {
	ll = 8;
      }else if (ll<3) {
	ll = 3;
      }
      sprintf( fmt, "%%11.%df", 8 - ll );
      fprintf( flog, fmt, G[k] );
      k++;
    }
    fprintf( flog, "\\" );
    for (; j<neffects; j++) {
      if (G[irc(i,i)] && G[irc(j,j)]) {
	fprintf( flog, "%11.3f", G[irc(i,j)] / sqrt(G[irc(i,i)]*G[irc(j,j)]));
      }else {
	fprintf( flog, "       .   " );
      }
    }
    fprintf( flog, "\n" );
  }

  fprintf( flog, "\n" );
  PEvar = 0;
  nGeq0 = nPEmat = nPEdir = 0;
  ///////////////////
  //  for (i=0; i<nth; i++) E[i] = 0.0;
  //////////////////
  for (i=k=0; i<neffects; i++) {
    if (including_trait(i) && !G[irc(i,i)]) {
      Geq0[ nGeq0 ] = i;
      nGeq0++;
    }
    if (E[irc(i,i)]) {
      PEvar = 1;
      if (i<ntraits) {
        PEdir[i] = 1;
	PEdir_trait[nPEdir] = i;
	nPEdir++;
      }else {
        PEmat[i-ntraits] = 1;
	PEmat_trait[nPEmat] = i-ntraits;
	nPEmat++;
      }
    }
  }

  if (PEvar) {
    fprintf( flog, "V(PE) (correlations above diagonal) ...\n" );
    if (nmaternal) {
      for (i=0; i<ntraits; i++) fprintf( flog, "%11s", " " );
      fprintf( flog, "maternal\n" );
    }
    fprintf( flog, " " );
    for (i=0; i<ntraits; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
    for (i=0; i<nmaternal; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
    fprintf( flog, "\n" );

    for (i=0; i<neffects; i++) {
      x = E[irc(i,i)];
      if (x<0) x = -x;
      // count number of integer digits (store as itmp[i])
      j = 0;
      while( x >= 1 ) {
	j++;
	x *= .1;
      }
      itmp[i] = j; 
    }
    k = 0;
    for (i=0; i<neffects; i++) {
      for (j=0; j<=i; j++) {
	// determine number of integer digits required (ll) and set format
	ll = (itmp[i] + itmp[j]) / 2;
	if (ll>8) {
	  ll = 8;
	}else if (ll<3) {
	  ll = 3;
	}
	sprintf( fmt, "%%11.%df", 8 - ll );
	fprintf( flog, fmt, E[k] );
	k++;
      }
      fprintf( flog, "\\" );
      for (; j<neffects; j++) {
	if (E[irc(i,i)] && E[irc(j,j)]) {
	  fprintf( flog, "%11.3f", E[irc(i,j)] / sqrt(E[irc(i,i)]*E[irc(j,j)]));
	}else {
	  fprintf( flog, "       .   " );
	}
      }
      fprintf( flog, "\n" );
    }

    for (i=0; i<neh; i++) E2[i] = E[i];
    // Ensure zero covariances between Edir and Emat, otherwise model is too complicated
    for (i=0; i<nmaternal; i++) {
      for (j=0; j<i; j++) {
	jm = j + ntraits;
	if (E2[irc(i,jm)]) {
	  sprintf( msg, "non-zero PE covariance (%f) between Dir_%d and Mat_%d IS IGNORED !!\n",
		   E2[irc(i,jm)], i, j );
	  warning( msg );
	  E2[irc(i,j+ntraits)] = 0.0;
	}
      }
    }
    if (invert( E2, neffects ) < 0) {
      PDerrors = 1;
      warning( "!! V(PE) is not positive definite, which could cause negative EDCs.\n" );
      warning( "!!  Suspect traits/effects include:" );
      for (j=flag=0; j<ntraits; j++) {
	if (E2[irc(j,j)] < 0) {
	  flag = 1;
	  if (nmaternal) {
	    sprintf( msg, ", %s direct", trait_label[j] );
	  }else {
	    sprintf( msg, ", %s", trait_label[j] );
	  }
	  warning( msg );
	}
      }
      for (j=0; j<nmaternal; j++) {
	if (E2[irc(ntraits+j,ntraits+j)] < 0) {
	  flag = 1;
	  sprintf( msg, ", %s maternal", trait_label[j] );
	  warning( msg );
	}
      }
      if (!flag) {
	// a suspect trait list could not be identified
	warning( "  .... ? Sorry, I was not able to narrow it down for you.\n" );
      }
      warning( "\n" );
      pr( flog, "Inverted Covariance Matrix (PE effects)", E2, neh );
      warning( "\n" );
    }

  }else {
    fprintf( flog, "No Permanent Environment effects in the model!\n" );
  }

  fprintf( flog, "\n" );
  if (CGvar) {
    fprintf( flog, "Variance of random CG effects (Note: correlations above diagonal are ignored) ...\n" );
    fprintf( flog, " " );
    for (i=0; i<ntraits; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
    fprintf( flog, "\n" );
    for (i=0; i<ntraits; i++) {
      for (j=0; j<=i; j++) fprintf( flog, "%11.3f", CG[irc(i,j)] );
      fprintf( flog, "\\" );
      for (; j<ntraits; j++) {
	if (CG[irc(i,i)] && CG[irc(j,j)]) {
	  fprintf( flog, "%11.3f", CG[irc(i,j)] / sqrt(CG[irc(i,i)]*CG[irc(j,j)]));
	}else {
	  fprintf( flog, "       .   " );
	}
      }
      // ignore correlations
      for (j=0; j<i; j++) CG[irc(i,j)] = 0.0;
      fprintf( flog, "\n" );
    }
    if (invert( CG, ntraits ) < 0) {
      PDerrors = 1;
      warning( "!! V(CG) is not positive definite, which could cause negative EDCs.\n" );
      warning( "!!  Suspect traits/effects include:" );
      for (j=flag=0; j<ntraits; j++) {
	if (CG[irc(j,j)] < 0) {
	  flag = 1;
	  sprintf( msg, ", %s", trait_label[j] );
	  warning( msg );
	}
      }
      if (!flag) {
	// a suspect trait list could not be identified
	warning( "  .... ? Sorry, I was not able to narrow it down for you.\n" );
      }
      warning( "\n" );
      pr( flog, "Inverted Covariance Matrix (CG effects)", CG, nth );
      warning( "\n" );
    }
  }else {
    fprintf( flog, "CG effects treated as fixed.\n" );
  }

  fprintf( flog, "\n" );
  fprintf( flog, "V(y) (correlations above diagonal) ...\n" );
  fprintf( flog, " " );
  for (i=0; i<ntraits; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
  fprintf( flog, "\n" );

  for (i=0; i<ntraits; i++) {
    x = vary[irc(i,i)];
    if (x<0) x = -x;
    // count number of integer digits (store as itmp[i])
    j = 0;
    while( x >= 1 ) {
      j++;
      x *= .1;
    }
    itmp[i] = j; 
  }
  k = 0;
  for (i=0; i<ntraits; i++) {
    for (j=0; j<=i; j++) {
      // determine number of integer digits required (ll) and set format
      ll = (itmp[i] + itmp[j]) / 2;
      if (ll>8) {
	ll = 8;
      }else if (ll<3) {
	ll = 3;
      }
      sprintf( fmt, "%%11.%df", 8 - ll );
      fprintf( flog, fmt, vary[k] );
      k++;
    }
    fprintf( flog, "\\" );
    for (; j<ntraits; j++) {
      if (vary[irc(i,i)] && vary[irc(j,j)]) {
	fprintf( flog, "%11.3f", 
		 vary[irc(i,j)] / sqrt(vary[irc(i,i)]*vary[irc(j,j)]));
      }else {
	fprintf( flog, "       .   " );
      }
    }
    fprintf( flog, "\n" );
  }

  if (nmaternal) {
    fprintf( flog, "\n" );
    fprintf( flog, "V(g-total) (correlations above diagonal) ...\n" );
    fprintf( flog, " " );
    for (i=0; i<ntraits; i++) fprintf( flog, "%s", cfmt( trait_label[i], 11 ) );
    fprintf( flog, "\n" );
    
    for (i=0; i<ntraits; i++) {
      x = varg[irc(i,i)];
      if (x<0) x = -x;
      // count number of integer digits (store as itmp[i])
      j = 0;
      while( x >= 1 ) {
	j++;
	x *= .1;
      }
      itmp[i] = j; 
    }
    k = 0;
    for (i=0; i<ntraits; i++) {
      for (j=0; j<=i; j++) {
	// determine number of integer digits required (ll) and set format
	ll = (itmp[i] + itmp[j]) / 2;
	if (ll>8) {
	  ll = 8;
	}else if (ll<3) {
	  ll = 3;
	}
	sprintf( fmt, "%%11.%df", 8 - ll );
	fprintf( flog, fmt, varg[k] );
	k++;
      }
      fprintf( flog, "\\" );
      for (; j<ntraits; j++) {
	if (varg[irc(i,i)] && varg[irc(j,j)]) {
	  fprintf( flog, "%11.3f", 
		   varg[irc(i,j)] / sqrt(varg[irc(i,i)]*varg[irc(j,j)]));
	}else {
	  fprintf( flog, "       .   " );
	}
      }
      fprintf( flog, "\n" );
    }
  }

  if (PDerrors && !ignore_PD_check) {
    error( "\nAt least 1 negative-definite matrix was detected.\n" );
  }

  return;
}

//**********************************************************************
REAL take_REAL( char *c, int len ) {
  // extract a sub-string and convert it to REAL
  static char tmp[BSZ];
  strncpy( tmp, c, len );
  tmp[len] = 0;
  return atof( tmp );
}

//**********************************************************************
int take_int( char *c, int len ) {
  // extract a sub-string and convert it to INTEGER
  static char tmp[BSZ];
  strncpy( tmp, c, len );
  tmp[len] = 0;
  return atoi( tmp );
}

//**********************************************************************
void update_allocated( int bytes ) {
  // keep track of total allocated RAM, for monitoring purposes
  int mb, gb;
#ifndef WINDOWS
  allocated[1] += bytes;
  if ((gb = allocated[1] / Gbyte)) {
    allocated[0] += gb;
    allocated[1] %= Gbyte;
  }
  mb = allocated[0] * Kbyte + allocated[1] / Mbyte;
  while( mb >= RAMmonitor ) {
    fprintf( stdout, "\n***  " );
    if (RAMmonitor >= Kbyte) {
      fprintf( stdout, "%.1f Gbytes", ( (double) RAMmonitor) / Kbyte );
      RAMmonitor += (Kbyte>>1);  // increment by .5 Gbytes
    }else {
      fprintf( stdout, "%d Mbytes", RAMmonitor );
      RAMmonitor *= 2;  // increment by doubling
    }
    fprintf( stdout, " of RAM was needed so far." );
    fprintf( stdout, "  ***\n\n" );
  }
#endif
  return;
}

//**********************************************************************
void warning( char *mmm ) {
  fprintf( stderr, "%s", mmm );
  if (flog) fprintf( flog, "%s", mmm );
  return;
}

//**********************************************************************
void *zinit( char *tmsg, int i, int j ) {
  // allocate RAM and initialize as zero
  void *p=0;
  if (!j) j = 1; // avoid allocation request for zero bytes
  if (i*j != 0) {
    p = calloc( i, j );
    if (p == NULL) {
      PRINTMEM;
      sprintf( msg, "insufficient RAM available for %s[%d*%d], program stopped!!\n",
	       tmsg, i, j );
      error( msg );
    }
    update_allocated( i*j );
  }else {
    if (i*j < i | i*j < j) {
      sprintf( msg, "zinit integer overflow with %d * %d for %s\n", i, j, tmsg );
      error( msg );
    }else {
      sprintf( msg, "zinit 0 bytes for %s !\n", tmsg );
      warning( msg );
    }
  }
  return p;
}

//**********************************************************************
int main( int argc, char *argv[] ) {
  int i, j, k, opt, flag;
  FILE *ff;
  char *host, *opt_list;

  Kbyte = 1<<10;
  Mbyte = 1<<20;
  Gbyte = 1<<30;
  RAMmonitor = RAMmonitor_0 = 1<<5; // start at 32 (Mbytes)

  setbuf( stdout, 0 );
  setbuf( stderr, 0 );

  z1 = 1.0 / 11.0;
  z2 = z1 * 2;
  z4 = z2 * 2;
  z8 = z4 * 2;
  z16 = z8 * 2;

  sprintf( RSS_tempfile, "crEDC_pid%d_RSS", getpid() );

  for (i=j=0; i<80; i++,j++) cmdline[j] = '*';
  cmdline[j] = '\n'; j++;
#ifndef WINDOWS
  if ((host = getenv( "HOSTNAME" ))) {
    sprintf( cmdline+j, "[%s] ", host ); j=strlen( cmdline );
  }
#endif
  strncpy( cmdline+j, "Command: ", 9 ); j+=9;
  for (i=0; i<argc; i++) {
    cmdline[j] = ' '; j++;
    k = strlen( argv[i] );
    strncpy( cmdline+j, argv[i], k ); j+=k;
  }
  cmdline[j] = '\n';  j++;
  for (i=0; i<80; i++,j++) cmdline[j] = '*';
  cmdline[j] = '\n';  j++;
  cmdline[j] = 0;
  tm1 = time( &tm0 );

  true_suff[0] = fname_true_rel[0] = 0;
  edc_pgm[0] = edc_log[0] = edc_par[0] = edc_tmp[0] = 0;
  flag = 1;

#ifdef IN_DEVELOPMENT
  sprintf( version, "%s-dev", VERSION );
  opt_list = "dhozp:t:w:ABCDEGHMNO:RSX";
#else
  opt_list = "dhop:w:AGHRS";
#ifdef RELIABILITY_MODULE
  sprintf( version, "%s-rel", VERSION );
#else
  sprintf( version, "%s", VERSION );
#endif
#endif

  while ((opt=getopt( argc, argv, opt_list )) != -1) {
    switch(opt) {
    case 'd':
      delimited_output = 1;
      break;
    case 'h':
      print_help(0);
    case 'o':
      redirect_output = 1;
      break;
    case 'p':
      if (optarg != NULL) {
        strcpy( param_file, optarg );
        flag = 0;
      }
      break;
    case 't':
      sprintf( true_suff, "_%s", optarg );
      break;
    case 'z':
      ignore_resid_correlations = 1;
      break;
    case 'w':
      // First reliability request overrides all others
      if (default_WF || !RELIABILITIES) {
	if (!strcmp( optarg, "ND" )) {
	  wfactor = NPROG;
	}else if (!strcmp( optarg, "Ne" )) {
	  wfactor = EFFECTIVE_PROG;
	}else if (!strcmp( optarg, "ITB" )) {
	  wfactor = ITB2000;
	}else if (!strcmp( optarg, "PGS1" )) {
	  wfactor = PGS1;
#ifdef IN_DEVELOPMENT
	}else if (!strcmp( optarg, "PGS2" )) {
	  wfactor = PGS2;
	}else if (!strcmp( optarg, "PGS3" )) {
	  wfactor = PGS3;
#endif
	}else if (!strcmp( optarg, "PGS4" )) {
	  wfactor = PGS4;
#ifdef RELIABILITY_MODULE
	}else if (!strcmp( optarg, "RELm" )) {
	  wfactor = RELY;
	  meyer_absorb = 1;
	  strcpy( rel_lbl, optarg );
	}else if (!strcmp( optarg, "REL4" )) {
	  wfactor = RELY;
	  strcpy( rel_lbl, optarg );
	}else if (!strcmp( optarg, "REL5" )) {
	  wfactor = RELY;
	  advanced_mate_absorb = 1;
	  strcpy( rel_lbl, optarg );
	}else if (!strcmp( optarg, "ACCm" )) {
	  wfactor = RELY;
	  meyer_absorb = 1;
	  output_accuracies = 1;
	  strcpy( rel_lbl, optarg );
	}else if (!strcmp( optarg, "ACC4" )) {
	  wfactor = RELY;
	  output_accuracies = 1;
	  strcpy( rel_lbl, optarg );
	}else if (!strcmp( optarg, "ACC5" )) {
	  wfactor = RELY;
	  advanced_mate_absorb = 1;
	  output_accuracies = 1;
	  strcpy( rel_lbl, optarg );
#endif
	}else {
	  fprintf( stderr, "\ninvalid -w option: %s\n", optarg );
	  print_help(5);
	}
      }
      default_WF = 0;
      break;
    case 'A':
      edc_reliabilities = 1;
      break;
    case 'B':
      bigCG = 1;
      break;
    case 'C':
      cancel_check = 1;
      break;
    case 'D':
      ignore_dam_rel = 1;
      break;
    case 'E':
      call_edc_pgm = 1;
      break;
    case 'G':
      count_grandprogeny = 1;
      break;
    case 'H':
      include_header = 1;
      break;
    case 'N':
      ignore_ETs = 1;
      break;
    case 'M':
      maternal_only = 1;
      break;
    case 'O':
      process_option( optarg );
      break;
    case 'R':
      output_integer = 0;
      break;
    case 'S':
      check_SI = 1;
      break;
    case 'X':
      ignore_wts = 1;
      break;
    case '?':
      print_help(5);
    case 1:
      fprintf( stderr, "\ninvalid option: %c\n", opt );
      print_help(5);
    }
  }

  if (include_header && !delimited_output) {
    if (redirect_output) {
      sprintf( cmdline+strlen(cmdline), 
	       "          NOTE: Option -H was ignored.\n" );
      sprintf( cmdline+strlen(cmdline), 
	       "                This option is only available with delimited output (-d)\n" );
      for (i=0; i<80; i++) msg[i] = '*'; msg[i] = 0;
      sprintf( cmdline+strlen(cmdline), "%s\n", msg );
    }else {
      warning( "\n" );
      warning( "  NOTE: Option -H was ignored.\n" );
      warning( "        This option is only available with delimited output (-d)\n" );
      warning( "\n" );
    }
  }
  sprintf( cmdline+strlen(cmdline), "\n" );

  if (flag) {
    fprintf( stderr, "\nno parameter file was specified!!\n" );
    print_help(5);
  }

  fop_quiet = 0;
  open_files();
  fop_quiet = 1;

  PRINTMEM;
  printf( "read ped...\n" );
  read_ped();
  PRINTMEM;
  printf( "accum_Ne...\n" );
  accum_Ne();
  PRINTMEM;
  if (!check_SI) summarize_means_cg();


  if (!NDAU_NE && !PGS4_NE && !RELIABILITIES) {
    printf( "compute_Ro...\n" );
    for (pp=ped+1; pp<=ped+nanml; pp++) {
      compute_Ro();
    }
    PRINTMEM;
  }

  printf( "compute_EDCs\n" );
#ifdef IN_DEVELOPMENT
  if (call_edc_pgm) {
    ff = fop( "Zinput_files", "w" );
    if (ff == NULL) {
      sprintf( msg, "error creating Zinput_files !! \n" );
      error( msg );
    }
    fprintf( ff, "%s_srt\n%s\n%s\n%s\n%s\n",
	     fnames[DATAFILE], edc_log, edc_par, fnames[PEDFILE], edc_tmp );
    fclose( ff );
    system( edc_pgm );
    read_edcs();
  }else {
    compute_edcs();
  }
#else
  compute_edcs();
#endif
  PRINTMEM;

  if (!RELIABILITIES) {
    output_edcs();
    summarize_means( means_edc, MEAN_VARS-1, "if EDC > 0" );
    summarize_means( means_np, MEAN_VARS, "if NP > 0" );
    summarize_means( means, MEAN_VARS, "if EDC > 0 and NP > 0" );
  }
  PRINTMEM;
#ifdef IN_DEVELOPMENT
  if (runcheck && !cancel_check) {
    printf( "check EDCs\n" );
    check_edcs();
    PRINTMEM;
  }
#endif

  PRINTMEM;
  fprintf( stdout, "Program completed successfully.\n" );
  ex(0);

  exit(0);
}

