/*
  Routine to invert a symmetric matrix (can be negative definite)
  - requires lower triangle of the matrix, stored in a zero-based array 
  - same algorythm as in Fortran program by Karin Meyer (dkmwhf.f)
  Copyright... Pete Sullivan, 2011
*/


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#ifndef REAL
#define REAL double
#endif

#if (REAL == double)
REAL trivial=.000000000001;
#else
REAL trivial=.000001;
#endif

void invpr( REAL *a, int n, char *msg ) {
  int i;
  for (i=0; i<n*(n+1)/2; i++) printf( "%9.4f", a[i] );
  printf( " %s\n", msg );
  return;
}

/************************************************************************/
/* NOTE: function irc() can be used in the calling program to index     */
/*   array elements consistent with the structure assumed for inverse() */
/************************************************************************/

long long irc2( int si, int sj, long long max_valid, char *msg, int exit_code ) {
  /*  returns (LOWER) half-stored element for (row,col) */
  /* NOTE: arrays are assumed to be zero-based */
  long long i, j, k;
  i = (long long) si;
  j = (long long) sj;
  if (i>j) {
    k = (i*(i+1)/2 + j);
  }else {
    k = (j*(j+1)/2 + i);
  }
  if (max_valid && (k+1) > max_valid || k < 0) {
    if (*msg) fprintf( stderr, "%s: ", msg );
    fprintf( stderr, "irc2(%lld,%lld)=%lld ... > max(base 0) = %lld\n",
	     i, j, k, max_valid-1 );
    if (exit_code) {
      fprintf( stderr, "  EXIT with code=%d\n", exit_code );
      exit( exit_code );
    }
  }
  return k;
}

int irc( int i, int j ) {
  /*  returns (LOWER) half-stored element for (row,col) */
  /* NOTE: arrays are assumed to be zero-based */
  if (i>j) {
    return (i*(i+1)/2 + j);
  }else {
    return (j*(j+1)/2 + i);
  }
}

void from_irc( int ij, int *row, int *col ) {
  int i, d;
  for ( i=d=0; d<ij; i++, d+=(i+1) );
  (*row) = i;
  (*col) = ij - (d-i);
  return;
}

int invert( REAL *a, int n ) {
  // matrix 'a' is replaced by the inverse, and the rank is returned (-rank if 'a' is neg.definite)
  REAL *tmp, x, z, zero, max, maxabs;
  int *pivots, i, j, k, l, pivot, pivot_row, diag, rank, psd=1;

#ifdef MONITOR_INVERSIONS
  time_t t0, t1;
  int decile;
  decile = n / 20;
  if (decile < 10) decile = 0;
#endif

  zero = (REAL) 0;
  tmp = malloc( n*sizeof(REAL) );
  pivots = calloc( n, sizeof(int) );
  if (tmp == NULL || pivots == NULL) {
    fprintf( stderr, "not enough memory to invert matrix (order=%d)\n", n );
    exit(5);
  }
  pivot = rank = 0;
  while( pivot < n ) {

    /* find next pivot row */
    max = maxabs = diag = 0;
    for (i=0; i<n; i++) {
      if (!pivots[i]) {
	if (a[diag] > 0) {
	  x = a[diag];
	}else {
	  x = -a[diag];
	}
	if (x > maxabs) {
	  maxabs = x;
	  max = a[diag];
	  if (max < trivial) psd = 0;
	  pivot_row = i;
	  k = diag - i;
	}
      }
      diag += i+2;
    }

    /* process pivot row */
    if (maxabs > trivial) {
      /* extract off-diagonal elements of pivot row */
      for (i=0; i<pivot_row; i++) {
	tmp[i] = a[k];
	k++;
      }
      k += i+1;
      for (i++; i<n; i++) {
	tmp[i] = a[k];
	k += i+1;
      }
      /* absorb pivot row into all rows < pivot_row */
      x = ((REAL) 1.0) / max;
      k = 0;
      for (i=0; i<pivot_row; i++) {
	z = tmp[i] * x;
	if (z == zero) {
	  k += i+1;
	}else {
	  for (j=0; j<=i; j++) {
	    a[k] -= tmp[j]*z;
	    k++;
	  }
	}
      }
      /* now i == pivot_row */
      for (j=0; j<i; j++) {
	a[k] *= x;
	k++;
      }
      a[k] = -x;
      k++;
      /* now process rows > pivot_row */
      for (i++; i<n; i++) {
	z = tmp[i] * x;
	if (z == zero) {
	  k += i+1;
	}else {
	  for (j=0; j<pivot_row; j++) {
	    a[k] -= tmp[j] * z;
	    k++;
	  }
	  a[k] *= x;
	  k++;
	  if (z != zero) {
	    for (j++; j<=i; j++) {
	      a[k] -= tmp[j] * z;
	      k++;
	    }
	  }
	}
      }

      pivot++;
      pivots[pivot_row] = pivot;
      rank=pivot;

    }else {
      pivot = n;
    }

#ifdef MONITOR_INVERSIONS
    if (decile && !(pivot%decile)) {
      fprintf( stdout, "%2.0f%% (%d of %d rows) inverted (current diag = %f).\n",
	       (100.0 * pivot)/n, pivot, n, max );
    }
#endif

  }

  /* finish up */
  k = n * (n+1) / 2;
  for (i=0; i<k; i++) if (a[i]) a[i] = -a[i];
  if (rank < n) {
    j = 0;
    for (i=0; i<n; i++) {
      if (!pivots[i]) {
	/* zero out row and column for generalized inverse */
	for (k=0; k<=i; k++) a[j+k] = zero;
	k += j-1;
	for (l=i+1; l<n; l++) {
	  k += l;
	  a[k] = zero;
	}
      }
      j += i+1;
    }
  }

  free(tmp);
  free(pivots);

#ifdef MONITOR_INVERSIONS
  if (decile) fprintf( stdout, "Matrix rank = %d\n", rank );
#endif

  if (psd) {
    return rank;
  }else {
    return -rank;
  }
}

